<?php
/**
 * Reviews - Add reviews and rating to profiles.
 */
if ( !defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}


/**
 * Registers reviews admin stylesheet
 *
 */
function datebook_reviews_admin_styles() {
	wp_register_style( 'datebook_reviews_admin_styles', OPTIONS_FRAMEWORK_DIRECTORY . 'css/datebook-reviews-admin.css', array( 'dashicons' ) );
	wp_register_script( 'datebook_reviews_admin_scripts', OPTIONS_FRAMEWORK_DIRECTORY . 'js/datebook-reviews-admin.js', array( 'jquery', 'admin-comments' ), false, true );
}
add_action( 'admin_init', 'datebook_reviews_admin_styles', 20 );

/**
 * Enqueues catalog admin stylesheet
 *
 */
function datebook_reviews_admin_enqueue_styles() {
	wp_enqueue_style( 'datebook_reviews_admin_styles' );
	wp_localize_script( 'datebook_reviews_admin_scripts', 'reviews_object', array( 'showcomm' => esc_html__( 'Show more reviews' ), 'endcomm' => esc_html__( 'No more reviews found.' ) ) );
	wp_enqueue_script( 'datebook_reviews_admin_scripts' );
	wp_enqueue_script( 'datebook_reviews_scripts' );

	wp_enqueue_script( 'admin-comments' );
	enqueue_comment_hotkeys_js();

}
add_action( 'admin_enqueue_scripts', 'datebook_reviews_admin_enqueue_styles', 20 );



/**
 * Checks if review modification screen is being displayed
 *
 */
function datebook_reviews_mod_screen() {
	if ( is_admin() && isset( $_GET[ 'c' ] ) && isset( $_GET[ 'action' ] ) && (isset( $_GET[ 'edit_review' ] ) || isset( $_GET[ 'trash_review' ] ) || isset( $_GET[ 'delete_review' ] ) || isset( $_GET[ 'approve_review' ] ) || isset( $_GET[ 'spam_review' ] )) ) {
		return true;
	}
	return false;
}
function datebook_reviews_admin_screen() {
	if ( is_admin() && isset( $_GET[ 'page' ] ) && strpos( $_GET[ 'page' ], 'datebook_reviews_' ) !== false ) {
		return true;
	}
	return false;
}
if ( !function_exists( 'datebook_url_includes' ) ) {
	function datebook_url_includes( $string, $contains ) {
		if ( strpos( $string, $contains ) !== false ) {
			return true;
		}
		return false;
	}
}

/**
 * Includes folder
 */
if ( !class_exists( 'WP_List_Table' ) ) {
	require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
}
if ( !class_exists( 'WP_Comments_List_Table' ) ) {
	require_once( ABSPATH . 'wp-admin/includes/class-wp-comments-list-table.php' );
}

class Datebook_Reviews_List_Table extends WP_Comments_List_Table {

	/**
	 * Constructor
	 */
	public function __construct( $args = array() ) {
		global $post_id;

		$post_id = isset( $_REQUEST[ 'p' ] ) ? absint( $_REQUEST[ 'p' ] ) : 0;

		if ( get_option( 'show_avatars' ) )
			add_filter( 'comment_author', 'floated_admin_avatar' );
		$screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
		WP_List_Table::__construct( array(
			'plural'	 => 'reviews',
			'singular'	 => 'review',
			'ajax'		 => true,
			'screen'	 => isset( $args[ 'screen' ] ) ? $args[ 'screen' ] : $screen,
		) );
	}


	public function get_columns() {
		global $post_id;

		$columns = array();

		if ( $this->checkbox )
			$columns[ 'cb' ] = '<input type="checkbox" />';

		$columns[ 'author' ]	 = esc_html__( 'Author' );
		$columns[ 'rating' ]	 = esc_html__( 'Rating', 'datebook' );
		$columns[ 'comment' ]	 = esc_html__( 'Review', 'datebook' );

		if ( !$post_id ) {
			$columns[ 'response' ] = esc_html__( 'Review item', 'product-review' );
		}

		return $columns;
	}


	public function ajax_user_can() {
		return current_user_can( 'edit_post' );
	}


	public function prepare_items() {
		global $post_id, $comment_status, $search, $comment_type;

		$comment_status	 = isset( $_REQUEST[ 'comment_status' ] ) ? sanitize_text_field( $_REQUEST[ 'comment_status' ] ) : 'all';
		if ( !in_array( $comment_status, array( 'all', 'moderated', 'approved', 'spam', 'trash' ) ) )
			$comment_status	 = 'all';

		$search = ( isset( $_REQUEST[ 's' ] ) ) ? sanitize_text_field( $_REQUEST[ 's' ] ) : '';

		$post_type = ( isset( $_REQUEST[ 'post_type' ] ) ) ? sanitize_key( $_REQUEST[ 'post_type' ] ) : '';

		$user_id = ( isset( $_REQUEST[ 'user_id' ] ) ) ? absint( $_REQUEST[ 'user_id' ] ) : '';

		$orderby = ( isset( $_REQUEST[ 'orderby' ] ) ) ? sanitize_text_field( $_REQUEST[ 'orderby' ] ) : '';
		$order	 = ( isset( $_REQUEST[ 'order' ] ) ) ? sanitize_text_field( $_REQUEST[ 'order' ] ) : '';

		$comments_per_page = $this->get_per_page( $comment_status );

		$doing_ajax = defined( 'DOING_AJAX' ) && DOING_AJAX;

		if ( isset( $_REQUEST[ 'number' ] ) ) {
			$number = (int) $_REQUEST[ 'number' ];
		} else {
			$number = $comments_per_page + min( 8, $comments_per_page ); // Grab a few extra
		}

		$page = $this->get_pagenum();

		if ( isset( $_REQUEST[ 'start' ] ) ) {
			$start = intval( $_REQUEST[ 'start' ] );
		} else {
			$start = ( $page - 1 ) * $comments_per_page;
		}

		if ( $doing_ajax && isset( $_REQUEST[ 'offset' ] ) ) {
			$start += intval( $_REQUEST[ 'offset' ] );
		}

		$status_map = array(
			'moderated'	 => 'hold',
			'approved'	 => 'approve',
			'all'		 => '',
		);

		$args		 = array(
			'status'	 => isset( $status_map[ $comment_status ] ) ? $status_map[ $comment_status ] : $comment_status,
			'search'	 => $search,
			'user_id'	 => $user_id,
			'offset'	 => $start,
			'number'	 => $number,
			'post_id'	 => $post_id,
			'orderby'	 => $orderby,
			'order'		 => $order,
			'post_type'	 => $post_type,
			'meta_key' => 'comment_type',
			'meta_value' => DATEBOOK_REVIEW_TYPE,
		);
		remove_all_filters( 'comments_clauses' );
		$_comments	 = get_comments( $args );

		if ( is_array( $_comments ) ) {
			update_comment_cache( $_comments );

			$this->items		 = array_slice( $_comments, 0, $comments_per_page );
			$this->extra_items	 = array_slice( $_comments, $comments_per_page );

			$_comment_post_ids = array_unique( wp_list_pluck( $_comments, 'comment_post_ID' ) );

			$this->pending_count = get_pending_comments_num( $_comment_post_ids );
		}

		$total_comments = get_comments( array_merge( $args, array(
			'count'	 => true,
			'offset' => 0,
			'number' => 0
		) ) );

		$this->set_pagination_args( array(
			'total_items'	 => $total_comments,
			'per_page'		 => $comments_per_page,
		) );
	}


	protected function get_views() {

		global $post_id, $comment_status, $comment_type;

		$status_links	 = array();
		$num_comments	 = ( $post_id ) ? wp_count_comments( $post_id ) : wp_count_comments();
		//, number_format_i18n($num_comments->moderated) ), "<span class='comment-count'>" . number_format_i18n($num_comments->moderated) . "</span>"),
		//, number_format_i18n($num_comments->spam) ), "<span class='spam-comment-count'>" . number_format_i18n($num_comments->spam) . "</span>")
		$stati			 = array(
			/* translators: %s: all comments count */
			'all' => _nx_noop(
			'All <span class="count">(%s)</span>', 'All <span class="count">(%s)</span>', 'comments'
			), // singular not used

			/* translators: %s: pending comments count */
			'moderated'	 => _nx_noop(
			'Pending <span class="count">(%s)</span>', 'Pending <span class="count">(%s)</span>', 'comments'
			),
			/* translators: %s: approved comments count */
			'approved'	 => _nx_noop(
			'Approved <span class="count">(%s)</span>', 'Approved <span class="count">(%s)</span>', 'comments'
			),
			/* translators: %s: spam comments count */
			'spam'		 => _nx_noop(
			'Spam <span class="count">(%s)</span>', 'Spam <span class="count">(%s)</span>', 'comments'
			),
			/* translators: %s: trashed comments count */
			'trash'		 => _nx_noop(
			'Trash <span class="count">(%s)</span>', 'Trash <span class="count">(%s)</span>', 'comments'
			)
		);

		if ( !EMPTY_TRASH_DAYS ) {
			unset( $stati[ 'trash' ] );
		}
		if ( isset( $_GET[ 'post_type' ] ) ) {
			$post_type	 = sanitize_text_field( $_GET[ 'post_type' ] );
			$page		 = 'edit.php?post_type=' . $post_type;
		} else {
			$post_type	 = 'post';
			$page		 = 'edit.php';
		}
		$link = $page . '&page=datebook_reviews_' . $post_type;
		if ( !empty( $comment_type ) && $comment_type != 'all' ) {
			$link = add_query_arg( 'comment_type', $comment_type, $link );
		}

		foreach ( $stati as $status => $label ) {

			$class = ( $status == $comment_status ) ? ' class="current"' : '';

			if ( !isset( $num_comments->$status ) ){
				$num_comments->$status = 10;
			}
			$link = add_query_arg( 'comment_status', $status, $link );
			if ( $post_id ){
				$link = add_query_arg( 'p', absint( $post_id ), $link );
			}
			
			if ( !empty( $_REQUEST['s'] ) ){
				$link = add_query_arg( 's', esc_attr( wp_unslash( $_REQUEST['s'] ) ), $link );
			}

			$status_links[ $status ] = "<a href='$link'$class>" . sprintf(translate_nooped_plural( $label, $num_comments->$status ), sprintf( '<span class="%s-count">%s</span>', ( $status === 'moderated' ) ? 'pending' : $status, number_format_i18n( $num_comments->$status ))) . '</a>';
		}

		/**
		 * Filter the comment status links with 'All', 'Pending', 'Approved', 'Spam', and 'Trash'
		 */
		return apply_filters( 'comment_status_links', $status_links );
	}


	public function display() {

		wp_nonce_field( "fetch-list-" . get_class( $this ), '_ajax_fetch_list_nonce' );
		$this->display_tablenav( 'top' );
		?>
		<table class="wp-list-table <?php echo implode( ' ', $this->get_table_classes() ); ?>">
			<thead>
				<tr>
					<?php $this->print_column_headers(); ?>
				</tr>
			</thead>

			<tbody id="the-comment-list" data-wp-lists="list:comment">
				<?php $this->display_rows_or_placeholder(); ?>
			</tbody>

			<tbody id="the-extra-comment-list" data-wp-lists="list:comment" style="display: none;">
				<?php
				$this->items = $this->extra_items;
				$this->display_rows();
				?>
			</tbody>

			<tfoot>
				<tr>
					<?php $this->print_column_headers( false ); ?>
				</tr>
			</tfoot>

		</table>

		<?php

		$this->display_tablenav( 'bottom' );

	}

	/**
	 * Display a comment count bubble
	 */
	protected function comments_bubble( $post_id, $pending_comments ) {
		$approved_comments = get_comments_number();

		$approved_comments_number	 = number_format_i18n( $approved_comments );
		$pending_comments_number	 = number_format_i18n( $pending_comments );

		$approved_only_phrase	 = sprintf( _n( '%s review', '%s reviews', $approved_comments ), $approved_comments_number );
		$approved_phrase		 = sprintf( _n( '%s approved review', '%s approved reviews', $approved_comments ), $approved_comments_number );
		$pending_phrase			 = sprintf( _n( '%s pending review', '%s pending reviews', $pending_comments ), $pending_comments_number );

		// No reviews at all.
		if ( !$approved_comments && !$pending_comments ) {
			printf( '<span aria-hidden="true">—</span><span class="screen-reader-text">%s</span>', esc_html__( 'No reviews' )
			);
			// Approved comments have different display depending on some conditions.
		} elseif ( $approved_comments ) {
			printf( '<a href="%s" class="post-com-count post-com-count-approved"><span class="comment-count-approved" aria-hidden="true">%s</span><span class="screen-reader-text">%s</span></a>', esc_url( add_query_arg( array( 'p' => $post_id, 'comment_status' => 'approved' ), admin_url( 'edit.php?post_type=' . DATEBOOK_POST_TYPE . '&page=datebook_reviews_' . DATEBOOK_POST_TYPE ) ) ), $approved_comments_number, $pending_comments ? $approved_phrase : $approved_only_phrase
			);
		} else {
			printf( '<span class="post-com-count post-com-count-no-comments"><span class="comment-count comment-count-no-comments" aria-hidden="true">%s</span><span class="screen-reader-text">%s</span></span>', $approved_comments_number, $pending_comments ? esc_html__( 'No approved reviews' ) : esc_html__( 'No reviews' )
			);
		}

		if ( $pending_comments ) {
			printf( '<a href="%s" class="post-com-count post-com-count-pending"><span class="comment-count-pending" aria-hidden="true">%s</span><span class="screen-reader-text">%s</span></a>', esc_url( add_query_arg( array( 'p' => $post_id, 'comment_status' => 'moderated' ), admin_url( 'edit.php?post_type=' . DATEBOOK_POST_TYPE . '&page=datebook_reviews_' . DATEBOOK_POST_TYPE ) ) ), $pending_comments_number, $pending_phrase
			);
		}
	}


	public function column_response( $comment ) {
		$post = get_post();

		if ( !$post ) {
			return;
		}

		if ( isset( $this->pending_count[ $post->ID ] ) ) {
			$pending_comments = $this->pending_count[ $post->ID ];
		} else {
			$_pending_count_temp				 = get_pending_comments_num( array( $post->ID ) );
			$pending_comments					 = $this->pending_count[ $post->ID ]	 = $_pending_count_temp[ $post->ID ];
		}

		if ( current_user_can( 'edit_post', $post->ID ) ) {
			$post_link = "<a href='" . get_edit_post_link( $post->ID ) . "' class='comments-edit-item-link'>";
			$post_link .= esc_html( get_the_title( $post->ID ) ) . '</a>';
		} else {
			$post_link = esc_html( get_the_title( $post->ID ) );
		}

		echo '<div class="response-links">';
		if ( 'attachment' == $post->post_type && ( $thumb = wp_get_attachment_image( $post->ID, array( 80, 60 ), true ) ) ) {
			echo $thumb;
		}
		echo $post_link;
		$post_type_object = get_post_type_object( $post->post_type );
		echo "<a href='" . get_permalink( $post->ID ) . "' class='comments-view-item-link'>" . $post_type_object->labels->view_item . '</a>';
		echo '<span class="post-com-count-wrapper">';
		$this->comments_bubble( $post->ID, $pending_comments );
		echo '</span> ';
		echo '</div>';
	}


	public function column_author( $comment ) {
		global $comment_status;

		$author_url = get_comment_author_url( $comment );

		$author_url_display = untrailingslashit( preg_replace( '|^http(s)?://(www\.)?|i', '', $author_url ) );
		if ( strlen( $author_url_display ) > 50 ) {
			$author_url_display = wp_html_excerpt( $author_url_display, 49, '&hellip;' );
		}

		echo "<strong>";
		comment_author( $comment );
		echo '</strong><br />';
		if ( !empty( $author_url_display ) ) {
			printf( '<a href="%s">%s</a><br />', esc_url( $author_url ), esc_html( $author_url_display ) );
		}

		if ( current_user_can( 'moderate_comments' ) ) {
			if ( !empty( $comment->comment_author_email ) ) {
				/* This filter is documented in wp-includes/comment-template.php */
				$email = apply_filters( 'comment_email', $comment->comment_author_email, $comment );

				if ( !empty( $email ) && '@' !== $email ) {
					printf( '<a href="%1$s">%2$s</a><br />', esc_url( 'mailto:' . $email ), esc_html( $email ) );
				}
			}
		}
	}


	/**
	 * Generate the table navigation above or below the table
	 */
	protected function display_tablenav( $which ) {

		if ( 'top' == $which ){
			wp_nonce_field( 'bulk-' . $this->_args[ 'plural' ] );
		}

		?>

		<div class="tablenav <?php echo esc_attr( $which ); ?>">

			<div class="alignleft actions bulkactions">
				<?php $this->bulk_actions( $which ); ?>
			</div>
			<?php
			$this->extra_tablenav( $which );
			$this->pagination( $which );
			?>

			<br class="clear" />
		</div>

		<?php

	}

	/**
	 * extra_tablenav without comment types drop-down
	 */
	protected function extra_tablenav( $which ) {

		global $comment_status, $comment_type;

		?>

		<div class="alignleft actions">

			<?php

			if ( ( 'spam' == $comment_status || 'trash' == $comment_status ) && current_user_can( 'moderate_comments' ) ) {
				wp_nonce_field( 'bulk-destroy', '_destroy_nonce' );
				$title = ( 'spam' == $comment_status ) ? esc_attr__( 'Empty Spam' ) : esc_attr__( 'Empty Trash' );
				submit_button( $title, 'apply', 'delete_all', false );
			}

			/**
			 * Fires after the Filter submit button for comment types. $comment_status The comment status name. Default 'All'.
			 */
			do_action( 'manage_comments_nav', $comment_status );

			echo '</div>';

		}

	}

	/**
	 * Post Comments List Table class.
	 */
	class DateBook_Reviews_Post_List_Table extends Datebook_Reviews_List_Table {

		protected function get_column_info() {
			$comment_type = DATEBOOK_REVIEW_TYPE;
			return array(
				array(
					'author'	 => esc_html__( 'Author', 'datebook' ),
					'rating'	 => esc_html__( 'Rating', 'datebook' ),
					'comment'	 => esc_html__( 'Review', 'datebook' ),
				),
				array(),
				array(),
				$comment_type,
			);
		}


		public function column_rating( $comment ) {
			echo '<div style="text-align:center;font-weight:bold;">' . datebook_reviews_get_title( $comment->comment_ID ) . '</div>';
			echo datebook_reviews_show_rating( $comment->comment_ID );
		}


		protected function get_table_classes() {
			$classes	 = parent::get_table_classes();
			$classes[]	 = 'wp-list-table';
			$classes[]	 = 'comments-box';
			return $classes;
		}


		public function display( $output_empty = false ) {
			$singular = $this->_args[ 'singular' ];
			wp_nonce_field( "fetch-list-" . get_class( $this ), '_ajax_fetch_list_nonce' );
			?>
			<table class="<?php echo implode( ' ', $this->get_table_classes() ); ?>" style="display:none;">
				<tbody id="the-comment-list"<?php
					   if ( $singular ) {
						   echo " data-wp-lists='list:$singular'";
					   }
					   ?>>
						   <?php
						   if ( !$output_empty ) {
							   $this->display_rows_or_placeholder();
						   }
						   ?>
				</tbody>
			</table>
			<?php
		}


		public function get_per_page( $comment_status = false ) {
			return 10;
		}

	}

	

function datebook_reviews_register_menu() {

	$hook = add_submenu_page( 'edit.php?post_type=' . DATEBOOK_POST_TYPE, esc_html__( 'Reviews', 'datebook' ), esc_html__( 'Reviews', 'datebook' ), 'moderate_comments', 'datebook_reviews_' . DATEBOOK_POST_TYPE, 'datebook_admin_reviews_list' );

}
add_action( 'admin_menu', 'datebook_reviews_register_menu', 15 );



	/**
	 * Ajax handler for getting comments.
	 */
	function datebook_admin_ajax_get_reviews( $action ) {
		add_filter( 'gettext', 'datebook_reviews_labels', 10, 3 );

		global $wp_list_table, $post_id;
		if ( empty( $action ) )
			$action = 'get-comments';

		check_ajax_referer( $action );

		if ( empty( $post_id ) && !empty( $_REQUEST[ 'p' ] ) ) {
			$id		 = absint( $_REQUEST[ 'p' ] );
			if ( !empty( $id ) )
				$post_id = $id;
		}

		if ( empty( $post_id ) ) {
			wp_die( -1 );
		}

		$wp_list_table = new DateBook_Reviews_Post_List_Table( array( 'screen' => 'edit-comments' ) );


		if ( !current_user_can( 'edit_post', $post_id ) ) {
			wp_die( -1 );
		}

		$wp_list_table->prepare_items();

		if ( !$wp_list_table->has_items() )
			wp_die( 1 );

		$x = new WP_Ajax_Response();
		ob_start();
		foreach ( $wp_list_table->items as $comment ) {
			if ( !current_user_can( 'edit_comment', $comment->comment_ID ) ) {
				continue;
			}
			get_comment( $comment );
			$wp_list_table->single_row( $comment );
		}
		$comment_list_item = ob_get_clean();

		$x->add( array(
			'what'	 => 'comments',
			'data'	 => $comment_list_item
		) );
		$x->send();
	}
	add_action( 'wp_ajax_get_reviews', 'datebook_admin_ajax_get_reviews' );


	/**
	 * Exclude reviews (comments) on from showing in comment feeds
	 */
	function datebook_hide_reviews_from_feeds( $where, $wp_comment_query ) {
		global $wpdb;

		$where .= $wpdb->prepare( " AND comment_type != %s", DATEBOOK_REVIEW_TYPE );
		$where .= $wpdb->prepare( " AND comment_type != %s", DATEBOOK_REVIEW_TYPE . '_' . DATEBOOK_POST_TYPE );
		
		return $where;
	}
	add_filter( 'comment_feed_where', 'datebook_hide_reviews_from_feeds', 10, 2 );


	/**
	 * Returns active comment types array
	 */
	function datebook_reviews_get_active_comment_types() {
		$types = array( DATEBOOK_REVIEW_TYPE );
		$types[] = DATEBOOK_REVIEW_TYPE . '_' . DATEBOOK_POST_TYPE;
		return $types;
	}


	/**
	 * Remove product reviews from the wp_count_comments function
	 */
	function datebook_remove_reviews_in_comment_counts( $stats, $post_id ) {
		global $wpdb, $comment_type;

		$count = wp_cache_get( "comments-{$post_id}", 'counts' );
		if ( false !== $count ) {
			return $count;
		}

		$post_id = (int) $post_id;

		$stats = wp_cache_get( "comments-{$post_id}", 'counts' );

		if ( false !== $stats ) {
			return $stats;
		}
		$where					 = 'WHERE';
		$active_comment_types	 = datebook_reviews_get_active_comment_types();
		foreach ( $active_comment_types as $key => $active_comment_type ) {
			if ( !empty( $key ) ) {
				$where .= ' AND';
			}
			if ( $comment_type == $active_comment_type ) {
				$where .= ' comment_type = "' . $active_comment_type . '"';
			} else {
				$where .= ' comment_type != "' . $active_comment_type . '"';
			}
		}

		if ( $post_id > 0 ) {
			$where .= $wpdb->prepare( " AND comment_post_ID = %d", $post_id );
		}

		$totals = (array) $wpdb->get_results( "
		SELECT comment_approved, COUNT( * ) AS total
		FROM {$wpdb->comments}
		{$where}
		GROUP BY comment_approved
	", ARRAY_A );

		$comment_count = array(
			'approved'				 => 0,
			'awaiting_moderation'	 => 0,
			'spam'					 => 0,
			'trash'					 => 0,
			'post-trashed'			 => 0,
			'total_comments'		 => 0,
			'all'					 => 0,
		);

		foreach ( $totals as $row ) {
			switch ( $row[ 'comment_approved' ] ) {
				case 'trash':
					$comment_count[ 'trash' ]				 = $row[ 'total' ];
					break;
				case 'post-trashed':
					$comment_count[ 'post-trashed' ]		 = $row[ 'total' ];
					break;
				case 'spam':
					$comment_count[ 'spam' ]				 = $row[ 'total' ];
					$comment_count[ 'total_comments' ] += $row[ 'total' ];
					break;
				case '1':
					$comment_count[ 'approved' ]			 = $row[ 'total' ];
					$comment_count[ 'total_comments' ] += $row[ 'total' ];
					$comment_count[ 'all' ] += $row[ 'total' ];
					break;
				case '0':
					$comment_count[ 'awaiting_moderation' ]	 = $row[ 'total' ];
					$comment_count[ 'total_comments' ] += $row[ 'total' ];
					$comment_count[ 'all' ] += $row[ 'total' ];
					break;
				default:
					break;
			}
		}
		$stats					 = $comment_count;
		$stats[ 'moderated' ]	 = $stats[ 'awaiting_moderation' ];
		unset( $stats[ 'awaiting_moderation' ] );

		$stats_object = (object) $stats;
		wp_cache_set( "comments-{$post_id}", $stats_object, 'counts' );

		return $stats_object;
	}
	add_filter( 'wp_count_comments', 'datebook_remove_reviews_in_comment_counts', 10, 2 );



/**
 * Output 'undo move to trash' text for reviews
 */
function datebook_reviews_trashnotice() {
	?>
	<div class="hidden" id="trash-undo-holder">
		<div class="trash-undo-inside"><?php printf( esc_html__( 'Review by %s moved to the trash.', 'datebook' ), '<strong></strong>' ); ?> <span class="undo untrash"><a href="#"><?php esc_html_e( 'Undo' ); ?></a></span></div>
	</div>
	<div class="hidden" id="spam-undo-holder">
		<div class="spam-undo-inside"><?php printf( esc_html__( 'Review by %s marked as spam.', 'datebook' ), '<strong></strong>' ); ?> <span class="undo unspam"><a href="#"><?php esc_html_e( 'Undo' ); ?></a></span></div>
	</div>
	<?php
}


/**
 * Handles manual comment update in admin
 */
function datebook_reviews_update_save( $comment_ID, $comment ) {
	if ( datebook_url_includes( $comment[ 'comment_type' ], DATEBOOK_REVIEW_TYPE ) ) {
		datebook_reviews_save_rating( $comment_ID );
	}
}
add_action( 'edit_comment', 'datebook_reviews_update_save', 10, 2 );


function datebook_reviews_row_actions( $actions, $comment ) {

	$actions[ 'edit' ] = '<a href="' . esc_url( admin_url( "comment.php?action=editcomment&c=$comment->comment_ID&edit_review=$comment->comment_ID" ) ) . '" aria-label="' . esc_html__( 'Edit this review' ) . '">' . esc_html__( 'Edit' ) . '</a>';

	return $actions;
}


function datebook_reviews_edit_screen( $review_id ) {
	if ( !$comment = get_comment( $review_id ) ) {
		comment_footer_die( esc_html__( 'Invalid review ID.' ) . sprintf( ' <a href="%s">' . esc_html__( 'Go back' ) . '</a>.', 'javascript:history.go(-1)' ) );
	}

	if ( !current_user_can( 'edit_comment', $review_id ) ) {
		comment_footer_die( esc_html__( 'Sorry, you are not allowed to edit this review.' ) );
	}

	if ( 'trash' == $comment->comment_approved ) {
		comment_footer_die( esc_html__( 'This review is in the Trash. Please move it out of the Trash if you want to edit it.' ) );
	}

	$comment = get_comment_to_edit( $review_id );
	include( ABSPATH . 'wp-admin/edit-form-comment.php' );
}




/**
 * Adds comment_type parameter to the url
 */
function datebook_reviews_add_submenu_param() {
	if ( datebook_reviews_admin_screen() && !isset( $_GET[ 'comment_type' ] ) ) {
		$comment_type	 = DATEBOOK_REVIEW_TYPE;
		$url			 = esc_url_raw( add_query_arg( 'comment_type', $comment_type ) );
		wp_redirect( $url );
		exit;
	}
}
add_action( 'admin_init', 'datebook_reviews_add_submenu_param' );


/**
 * Changes comment to review on review edit screen
 */
function datebook_reviews_edit_title( $translated, $text, $textdomain ) {
	if ( datebook_reviews_mod_screen() ) {
		if ( strpos( $text, 'Edit Comment' ) !== false ) {
			$translated = str_replace( 'Edit Comment', esc_html__( 'Edit Review', 'datebook' ), $text );
		} else if ( strpos( $text, 'Moderate Comment' ) !== false ) {
			$translated = str_replace( 'Moderate Comment', esc_html__( 'Moderate Review', 'datebook' ), $text );
		} else if ( strpos( $text, 'You are about to approve the following comment:' ) !== false ) {
			$translated = str_replace( 'You are about to approve the following comment:', esc_html__( 'You are about to approve the following review:', 'datebook' ), $text );
		} else if ( strpos( $text, 'Approve Comment' ) !== false ) {
			$translated = str_replace( 'Approve Comment', esc_html__( 'Approve Review', 'datebook' ), $text );
		} else if ( strpos( $text, 'In Response To' ) !== false ) {
			$translated = str_replace( 'In Response To', esc_html__( 'Reviewed', 'datebook' ), $text );
		}
	}
	return $translated;
}
add_filter( 'gettext', 'datebook_reviews_edit_title', 10, 3 );


function datebook_reviews_highlight_edit_menu( $parent_file = null ) {
	if ( datebook_reviews_mod_screen() ) {
		global $parent_file, $submenu_file;
		$comment_id		 = intval( $_GET[ 'c' ] );
		$comment		 = get_comment( $comment_id );
		$post_type		 = get_post_type( $comment->comment_post_ID );
		$parent_file	 = 'edit.php?post_type=' . $post_type;
		$submenu_file	 = 'datebook_reviews_' . $post_type;
	}
	return $parent_file;
}
//add_action( 'in_admin_header', 'datebook_reviews_highlight_edit_menu' );
add_filter( 'parent_file', 'datebook_reviews_highlight_edit_menu' );


function datebook_reviews_columns( $columns ) {
	$new_columns = array(
		'cb'		 => isset( $columns[ 'cb' ] ) ? $columns[ 'cb' ] : '',
		'author'	 => isset( $columns[ 'author' ] ) ? $columns[ 'author' ] : '',
		'rating'	 => esc_html__( 'Rating', 'datebook' ),
		'comment'	 => isset( $columns[ 'comment' ] ) ? $columns[ 'comment' ] : '',
		'response'	 => isset( $columns[ 'response' ] ) ? $columns[ 'response' ] : '',
	);
	$columns	 = $new_columns;

	return $columns;
}


function datebook_reviews_inititalize_plus_columns() {
	if ( function_exists( 'get_current_screen' ) && datebook_reviews_admin_screen() ) {
		$screen = get_current_screen();
		if ( isset( $screen->id ) ) {
			add_filter( 'manage_' . $screen->id . '_columns', 'datebook_reviews_columns', 99 );
		}
	}
}
add_action( 'current_screen', 'datebook_reviews_inititalize_plus_columns' );



function datebook_reviews_ajax_columns( $columns ) {
	if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
		if ( isset( $_POST[ 'comment_ID' ] ) ) {
			$comment_id = intval( $_POST[ 'comment_ID' ] );
			if ( !empty( $comment_id ) ) {
				$comment_type = get_comment_type( $comment_id );
				if ( datebook_url_includes( $comment_type, DATEBOOK_REVIEW_TYPE ) ) {
					$new_columns = array(
						'cb'		 => $columns[ 'cb' ],
						'author'	 => $columns[ 'author' ],
						'rating'	 => esc_html__( 'Rating', 'datebook' ),
						'comment'	 => $columns[ 'comment' ],
						'response'	 => $columns[ 'response' ]
					);
					$columns	 = $new_columns;
				}
			}
		}
	}
	return $columns;
}
add_filter( 'manage_edit-comments_columns', 'datebook_reviews_ajax_columns' );


/**
 * Returns revuew title
 */
function datebook_reviews_get_title( $review_id ) {
	$title = empty( $review_id ) ? '' : get_comment_meta( $review_id, 'datebook_review_title', true );
	return $title;
}


	/**
	 * Return product review value
	 */
	function datebook_reviews_get_rating( $review_id ) {
		$rating = empty( $review_id ) ? 0 : intval( get_comment_meta( $review_id, 'review_rating', true ) );
		return $rating;
	}

	/**
	 * Returns review rating HTML
	 */
	function datebook_reviews_show_rating( $review_id = 0, $editable = null, $label = null, $rating = null ) {

		if ( $rating === null ) {
			$rating = empty( $review_id ) ? 0 : datebook_reviews_get_rating( $review_id );
		}
		
		if ( !empty( $editable ) ) {
			$return = '<p class="mb-2 datebook-review-rating allow-edit"><input type="hidden" name="datebook_review_rating" value="' . $rating . '">';
		} else {
			$return = '<p class="mb-2 datebook-review-rating" data-current_rating="' . $rating . '">';
		}
		if ( !empty( $label ) ) {
			$return .= '<label class="rating-label" for="datebook_review_rating">' . $label . '</label>';
		}
		for ( $i = 1; $i <= $rating; $i++ ) {
			$return .= '<span class="datebook-rate-star rating-on rate-' . $i . '" data-rating="' . $i . '"></span>';
		}
		$rounded_rating = round( $rating );
		if ( $rounded_rating !== $rating ) {
			if ( abs( $rounded_rating - $rating ) > 0.2 ) {
				$return .= '<span class="datebook-rate-star rating-off rate-half" data-rating="' . $i . '"></span>';
			} else {
				$rating = floor( $rating );
			}
		}
		$off_rating = 5 - $rating;
		for ( $i = 1; $i <= $off_rating; $i++ ) {
			$a		 = $i + $rating;
			$return	 .= '<span class="datebook-rate-star rating-off rate-' . $a . '" data-rating="' . $a . '"></span>';
		}
		
		$return .= '</p>';
		return $return;
	}


function datebook_reviews_column( $column, $review_id ) {
	if ( datebook_reviews_admin_screen() || (defined( 'DOING_AJAX' ) && DOING_AJAX) ) {
		if ( 'rating' == $column ) {
			echo '<div class="review-title">' . datebook_reviews_get_title( $review_id ) . '</div>';
			echo datebook_reviews_show_rating( $review_id );
		}
	}
}
add_filter( 'manage_comments_custom_column', 'datebook_reviews_column', 10, 2 );


/**
 * Change comments to reviews text on reviews admin page
 */
function datebook_reviews_labels( $translated, $text, $textdomain ) {
	if ( is_admin() && $textdomain == 'default' && datebook_reviews_admin_screen() ) {
		if ( strpos( $text, 'comments' ) !== false ) {
			$translated = str_replace( 'comments', esc_html__( 'reviews', 'datebook' ), $text );
		} else if ( strpos( $text, 'Comments' ) !== false ) {
			$translated = str_replace( 'Comments', esc_html__( 'Reviews', 'datebook' ), $text );
		} else if ( strpos( $text, 'comment' ) !== false ) {
			$translated = str_replace( 'comment', esc_html__( 'review', 'datebook' ), $text );
		} else if ( strpos( $text, 'Comment' ) !== false ) {
			$translated = str_replace( 'Comment', esc_html__( 'Review', 'datebook' ), $text );
		}
	}
	return $translated;
}


/**
 * Checks if review post type admin screen is being displayed
 *
 * @return boolean
 */
function datebook_reviews_post_type_screen() {
	if ( isset( $_GET[ 'post_type' ] ) && $_GET[ 'post_type' ] == DATEBOOK_POST_TYPE ) {
		return true;
	}
	return false;
}


/**
 * Change comments to reviews text on reviews admin page
 */
function datebook_reviews_force_labels( $translated, $text, $textdomain ) {
	if ( is_admin() && $textdomain == 'default' ) {
		if ( strpos( $text, 'comments' ) !== false ) {
			$translated = str_replace( 'comments', esc_html__( 'reviews', 'datebook' ), $text );
		} else if ( strpos( $text, 'Comments' ) !== false ) {
			$translated = str_replace( 'Comments', esc_html__( 'Reviews', 'datebook' ), $text );
		} else if ( strpos( $text, 'comment' ) !== false ) {
			$translated = str_replace( 'comment', esc_html__( 'review', 'datebook' ), $text );
		} else if ( strpos( $text, 'Comment' ) !== false ) {
			$translated = str_replace( 'Comment', esc_html__( 'Review', 'datebook' ), $text );
		}
	}
	return $translated;
}


function datebook_reviews_replace_comments_labels() {
	if ( datebook_reviews_post_type_screen() ) {
		add_filter( 'gettext', 'datebook_reviews_force_labels', 10, 3 );
	}
}
add_action( 'admin_init', 'datebook_reviews_replace_comments_labels' );



function datebook_reviews_admin_process() {
	global $datebook_reviews_table, $wpdb;
	$doaction = $datebook_reviews_table->current_action();
	if ( $doaction ) {
		check_admin_referer( 'bulk-reviews' );

		if ( 'delete_all' == $doaction && !empty( $_REQUEST[ 'pagegen_timestamp' ] ) ) {
			$comment_status	 = wp_unslash( $_REQUEST[ 'comment_status' ] );
			$delete_time	 = wp_unslash( $_REQUEST[ 'pagegen_timestamp' ] );
			$comment_ids	 = $wpdb->get_col( $wpdb->prepare( "SELECT comment_ID FROM $wpdb->comments WHERE comment_approved = %s AND %s > comment_date_gmt", $comment_status, $delete_time ) );
			$doaction		 = 'delete';
		} elseif ( isset( $_REQUEST[ 'delete_comments' ] ) ) {
			$comment_ids = $_REQUEST[ 'delete_comments' ];
			$doaction	 = ( $_REQUEST[ 'action' ] != -1 ) ? $_REQUEST[ 'action' ] : $_REQUEST[ 'action2' ];
		} elseif ( isset( $_REQUEST[ 'ids' ] ) ) {
			$comment_ids = array_map( 'absint', explode( ',', $_REQUEST[ 'ids' ] ) );
		} elseif ( wp_get_referer() ) {
			wp_safe_redirect( wp_get_referer() );
			exit;
		}

		$approved	 = $unapproved	 = $spammed	 = $unspammed	 = $trashed	 = $untrashed	 = $deleted	 = 0;

		$redirect_to = remove_query_arg( array( 'trashed', 'untrashed', 'deleted', 'spammed', 'unspammed', 'approved', 'unapproved', 'ids' ), wp_get_referer() );
		$redirect_to = add_query_arg( 'paged', $pagenum, $redirect_to );

		foreach ( $comment_ids as $comment_id ) { // Check the permissions on each
			if ( !current_user_can( 'edit_comment', $comment_id ) )
				continue;

			switch ( $doaction ) {
				case 'approve' :
					wp_set_comment_status( $comment_id, 'approve' );
					$approved++;
					break;
				case 'unapprove' :
					wp_set_comment_status( $comment_id, 'hold' );
					$unapproved++;
					break;
				case 'spam' :
					wp_spam_comment( $comment_id );
					$spammed++;
					break;
				case 'unspam' :
					wp_unspam_comment( $comment_id );
					$unspammed++;
					break;
				case 'trash' :
					wp_trash_comment( $comment_id );
					$trashed++;
					break;
				case 'untrash' :
					wp_untrash_comment( $comment_id );
					$untrashed++;
					break;
				case 'delete' :
					wp_delete_comment( $comment_id );
					$deleted++;
					break;
			}
		}

		if ( $approved )
			$redirect_to = add_query_arg( 'approved', $approved, $redirect_to );
		if ( $unapproved )

			$redirect_to = add_query_arg( 'unapproved', $unapproved, $redirect_to );
		if ( $spammed )
			$redirect_to = add_query_arg( 'spammed', $spammed, $redirect_to );
		if ( $unspammed )
			$redirect_to = add_query_arg( 'unspammed', $unspammed, $redirect_to );
		if ( $trashed )
			$redirect_to = add_query_arg( 'trashed', $trashed, $redirect_to );
		if ( $untrashed )
			$redirect_to = add_query_arg( 'untrashed', $untrashed, $redirect_to );
		if ( $deleted )
			$redirect_to = add_query_arg( 'deleted', $deleted, $redirect_to );
		if ( $trashed || $spammed )
			$redirect_to = add_query_arg( 'ids', join( ',', $comment_ids ), $redirect_to );

		wp_safe_redirect( $redirect_to );
		exit;
	} elseif ( !empty( $_GET[ '_wp_http_referer' ] ) ) {
		wp_redirect( remove_query_arg( array( '_wp_http_referer', '_wpnonce' ), wp_unslash( $_SERVER[ 'REQUEST_URI' ] ) ) );
		exit;
	}
}


function datebook_reviews_admin_init() {
	if ( datebook_reviews_admin_screen() ) {
		if ( isset( $_GET[ 'edit_review' ] ) ) {
			$review_id = intval( $_GET[ 'edit_review' ] );
			datebook_reviews_edit_screen( $review_id );
		} else {
			global $datebook_reviews_table;
			add_filter( 'comment_row_actions', 'datebook_reviews_row_actions', 10, 2 );
			$datebook_reviews_table = new Datebook_Reviews_List_Table( );
			add_action( 'admin_init', 'datebook_reviews_admin_process', 99 );
		}
	}
}
add_action( 'admin_init', 'datebook_reviews_admin_init', 98 );


function datebook_admin_reviews_list() {
	global $post_id, $comment_status, $search, $comment_type, $datebook_reviews_table;

	$comment_type = DATEBOOK_REVIEW_TYPE;
	wp_cache_delete( "comments-{$post_id}", 'counts' );
	add_filter( 'gettext', 'datebook_reviews_labels', 10, 3 );

	$pagenum = $datebook_reviews_table->get_pagenum();

	$datebook_reviews_table->prepare_items();

	if ( $post_id ){
		$title	 = sprintf( esc_html__( 'Reviews on &#8220;%s&#8221;' ), wp_html_excerpt( _draft_or_post_title( $post_id ), 50, '&hellip;' ) );
	} else {
		$title	 = esc_html__( 'Reviews' );
	}

	?>

	<div class="wrap">
		<h1><?php
			if ( $post_id ){
				echo sprintf( esc_html__( 'Reviews on &#8220;%s&#8221;' ), sprintf( '<a href="%s">%s</a>', get_edit_post_link( $post_id ), wp_html_excerpt( _draft_or_post_title( $post_id ), 50, '&hellip;' )));
			} else {
				esc_html_e( 'Reviews' );
			}

			if ( isset( $_REQUEST[ 's' ] ) && $_REQUEST[ 's' ] )
				echo '<span class="subtitle">' . sprintf( esc_html__( 'Search results for &#8220;%s&#8221;' ), wp_html_excerpt( esc_html( wp_unslash( $_REQUEST[ 's' ] ) ), 50, '&hellip;' ) ) . '</span>';
			?>
		</h1>

		<?php
		if ( isset( $_REQUEST[ 'error' ] ) ) {
			$error		 = (int) $_REQUEST[ 'error' ];
			$error_msg	 = '';
			switch ( $error ) {
				case 1 :
					$error_msg	 = esc_html__( 'Invalid review ID.' );
					break;
				case 2 :
					$error_msg	 = esc_html__( 'You are not allowed to edit reviews on this product.' );
					break;
			}
			if ( $error_msg )
				echo '<div id="moderated" class="error"><p>' . $error_msg . '</p></div>';
		}

		if ( isset( $_REQUEST[ 'approved' ] ) || isset( $_REQUEST[ 'deleted' ] ) || isset( $_REQUEST[ 'trashed' ] ) || isset( $_REQUEST[ 'untrashed' ] ) || isset( $_REQUEST[ 'spammed' ] ) || isset( $_REQUEST[ 'unspammed' ] ) || isset( $_REQUEST[ 'same' ] ) ) {
			$approved	 = isset( $_REQUEST[ 'approved' ] ) ? (int) $_REQUEST[ 'approved' ] : 0;
			$deleted	 = isset( $_REQUEST[ 'deleted' ] ) ? (int) $_REQUEST[ 'deleted' ] : 0;
			$trashed	 = isset( $_REQUEST[ 'trashed' ] ) ? (int) $_REQUEST[ 'trashed' ] : 0;
			$untrashed	 = isset( $_REQUEST[ 'untrashed' ] ) ? (int) $_REQUEST[ 'untrashed' ] : 0;
			$spammed	 = isset( $_REQUEST[ 'spammed' ] ) ? (int) $_REQUEST[ 'spammed' ] : 0;
			$unspammed	 = isset( $_REQUEST[ 'unspammed' ] ) ? (int) $_REQUEST[ 'unspammed' ] : 0;
			$same		 = isset( $_REQUEST[ 'same' ] ) ? (int) $_REQUEST[ 'same' ] : 0;

			if ( $approved > 0 || $deleted > 0 || $trashed > 0 || $untrashed > 0 || $spammed > 0 || $unspammed > 0 || $same > 0 ) {
				if ( $approved > 0 )
					$messages[] = sprintf( _n( '%s review approved', '%s reviews approved', $approved ), $approved );

				if ( $spammed > 0 ) {
					$ids		 = isset( $_REQUEST[ 'ids' ] ) ? $_REQUEST[ 'ids' ] : 0;
					$messages[]	 = sprintf( _n( '%s review marked as spam.', '%s reviews marked as spam.', $spammed ), $spammed ) . ' <a href="' . esc_url( wp_nonce_url( "edit.php?post_type=" . DATEBOOK_POST_TYPE . "&page=datebook_reviews_" . DATEBOOK_POST_TYPE . "&doaction=undo&action=unspam&ids=$ids", "bulk-comments" ) ) . '">' . esc_html__( 'Undo' ) . '</a><br />';
				}

				if ( $unspammed > 0 )
					$messages[] = sprintf( _n( '%s review restored from the spam', '%s reviews restored from the spam', $unspammed ), $unspammed );

				if ( $trashed > 0 ) {
					$ids		 = isset( $_REQUEST[ 'ids' ] ) ? $_REQUEST[ 'ids' ] : 0;
					$messages[]	 = sprintf( _n( '%s review moved to the Trash.', '%s reviews moved to the Trash.', $trashed ), $trashed ) . ' <a href="' . esc_url( wp_nonce_url( "edit.php?post_type=" . DATEBOOK_POST_TYPE . "&page=datebook_reviews_" . DATEBOOK_POST_TYPE . "&doaction=undo&action=untrash&ids=$ids", "bulk-comments" ) ) . '">' . esc_html__( 'Undo' ) . '</a><br />';
				}

				if ( $untrashed > 0 )
					$messages[] = sprintf( _n( '%s review restored from the Trash', '%s reviews restored from the Trash', $untrashed ), $untrashed );

				if ( $deleted > 0 )
					$messages[] = sprintf( _n( '%s review permanently deleted', '%s reviews permanently deleted', $deleted ), $deleted );

				if ( $same > 0 && $comment = get_comment( $same ) ) {
					switch ( $comment->comment_approved ) {
						case '1' :
							$messages[]	 = esc_html__( 'This review is already approved.' ) . ' <a href="' . esc_url( admin_url( "edit.php?post_type=" . DATEBOOK_POST_TYPE . "&page=datebook_reviews_" . DATEBOOK_POST_TYPE . "&edit_review=$same" ) ) . '">' . esc_html__( 'Edit review' ) . '</a>';
							break;
						case 'trash' :
							$messages[]	 = esc_html__( 'This review is already in the Trash.' ) . ' <a href="' . esc_url( admin_url( 'edit.php?post_type=' . DATEBOOK_POST_TYPE . '&page=datebook_reviews_' . DATEBOOK_POST_TYPE . '&comment_status=trash' ) ) . '"> ' . esc_html__( 'View Trash' ) . '</a>';
							break;
						case 'spam' :
							$messages[]	 = esc_html__( 'This review is already marked as spam.' ) . ' <a href="' . esc_url( admin_url( "edit.php?post_type=" . DATEBOOK_POST_TYPE . "&page=datebook_reviews_" . DATEBOOK_POST_TYPE . "&edit_review=$same" ) ) . '">' . esc_html__( 'Edit comment' ) . '</a>';
							break;
					}
				}

				echo '<div id="moderated" class="updated notice is-dismissible"><p>' . implode( "<br/>\n", $messages ) . '</p></div>';
			}
		}
		?>

		<?php $datebook_reviews_table->views(); ?>

		<form id="comments-form" method="get">

			<?php $datebook_reviews_table->search_box( esc_html__( 'Search Reviews' ), 'comment' ); ?>

			<?php if ( $post_id ) : ?>
				<input type="hidden" name="p" value="<?php echo esc_attr( intval( $post_id ) ); ?>" />
				<?php
			endif;

			if ( isset( $_GET[ 'post_type' ] ) ) {
				$post_type = esc_attr( $_GET[ 'post_type' ] );
			} else {
				$post_type = 'post';
			}
			?>
			<input type="hidden" name="comment_status" value="<?php echo esc_attr( $comment_status ); ?>" />
			<input type="hidden" name="pagegen_timestamp" value="<?php echo esc_attr( current_time( 'mysql', 1 ) ); ?>" />
			<input type="hidden" name="page" value="<?php echo esc_attr( $_GET[ 'page' ] ) ?>" />
			<input type="hidden" name="post_type" value="<?php echo $post_type ?>" />
			<input type="hidden" name="comment_type" value="<?php echo esc_attr( $_GET[ 'comment_type' ] ) ?>" />

			<input type="hidden" name="_total" value="<?php echo esc_attr( $datebook_reviews_table->get_pagination_arg( 'total_items' ) ); ?>" />
			<input type="hidden" name="_per_page" value="<?php echo esc_attr( $datebook_reviews_table->get_pagination_arg( 'per_page' ) ); ?>" />
			<input type="hidden" name="_page" value="<?php echo esc_attr( $datebook_reviews_table->get_pagination_arg( 'page' ) ); ?>" />

			<?php if ( isset( $_REQUEST[ 'paged' ] ) ) { ?>
				<input type="hidden" name="paged" value="<?php echo esc_attr( absint( $_REQUEST[ 'paged' ] ) ); ?>" />
			<?php } ?>

			<?php $datebook_reviews_table->display(); ?>
		</form>
	</div>

	<div id="ajax-response"></div>

	<?php
	wp_comment_reply( '-1', true, 'detail' );
	wp_comment_trashnotice();
}




add_action( 'wp_ajax_datebook_hide_translate_notice', 'ajax_hide_revs_translation_notice' );

/**
 * Handles ajax translation notice hide
 *
 */
function ajax_hide_revs_translation_notice() {
	datebook_reviews_plugin_translation_notice_hide();
	wp_die();
}

function datebook_reviews_plugin_translation_notice_hide() {
	set_site_transient( 'datebook_reviews_hide_plugin_translation_info', 1 );
}


