<?php
/**
 * Create custom fields for profile form.
 *
 * @class DateBook_Custom_Fields_Table
 * @author Pagick Themes
 * @since 1.6
 */
	 

/**
 * Define DateBook Table List
 * ============================================================================
 *
 * In this part you are going to define custom table list class,
 * that will display your database records in nice looking table
 *
 * http://codex.wordpress.org/Class_Reference/WP_List_Table
 * http://wordpress.org/extend/plugins/custom-list-table-example/
 */

if (!class_exists('WP_List_Table')) {

	require_once(ABSPATH . 'wp-admin/includes/class-wp-list-table.php');

}


/**
 * DateBook_Custom_Fields_Table class that will display our custom table
 * records in nice table
 */
class DateBook_Reserved_Custom_Fields_Table extends WP_List_Table {

	/**
	 * [REQUIRED] You must declare constructor and give some basic params
	 */
	function __construct() {

		global $status, $page;

		parent::__construct(array(
			'singular' => 'field',
			'plural' => 'fields',
		));
	}


	/**
	 * [REQUIRED] this is a default column renderer
	 *
	 * @param $item - row (key, value array)
	 * @param $column_name - string (key)
	 * @return HTML
	 */
	function column_default($item, $column_name) {
		
		if ($column_name == 'field_items'){
			
			// if foreign language do not exist, then show title in English
			$item[$column_name] = isset($item[$column_name . '_' . DATEBOOK_WPLANG]) ? $item[$column_name . '_' . DATEBOOK_WPLANG] : $item[$column_name . '_en_US'];

			return esc_attr($item[$column_name]);
			
		} else {
			
			return esc_attr($item[$column_name]);
			
		}
		
	}


	/**
	 * [OPTIONAL] this is example, how to render specific column
	 *
	 * method name must be like this: "column_[column_name]"
	 *
	 * @param $item - row (key, value array)
	 * @return HTML
	 */
	function column_searchable($item) {

		if (isset($item['searchable']) && $item['searchable'] == 1) {

			$out = '<span style="color:green">' . esc_html__('Yes', 'datebook') . '</span>';

		} else {

			$out = '';

		}

		return $out;

	}


	/**
	 * [OPTIONAL] this is example, how to render specific column
	 *
	 * method name must be like this: "column_[column_name]"
	 *
	 * @param $item - row (key, value array)
	 * @return HTML
	 */
	function column_required($item) {

		if (isset($item['required']) && $item['required'] == 1) {

			$out = '<span style="color:green">' . esc_html__('Yes', 'datebook') . '</span>';

		} else {

			$out = '';

		}

		return $out;

	}


	/**
	 * [OPTIONAL] this is example, how to render specific column
	 *
	 * method name must be like this: "column_[column_name]"
	 *
	 * @param $item - row (key, value array)
	 * @return HTML
	 */
	function column_visible_to_admin_only($item) {

		if (isset($item['visible_to_admin_only']) && $item['visible_to_admin_only'] == 1) {

			$out = '<span style="color:green">' . esc_html__('Yes', 'datebook') . '</span>';

		} else {

			$out = '';

		}

		return $out;

	}


	/**
	 * [OPTIONAL] this is example, how to render specific column
	 *
	 * method name must be like this: "column_[column_name]"
	 *
	 * @param $item - row (key, value array)
	 * @return HTML
	 */
	function column_active($item) {
		
		if (isset($item['active']) && $item['active'] == 1) {
			
			$out = '<span style="color:green">' . esc_html__('Yes', 'datebook') . '</span>';
			
		} else {
			
			$out = '';
			
		}

		return $out;

	}


	/**
	 * [OPTIONAL] this is example, how to render column with actions,
	 * when you hover row "Edit | Delete" links showed
	 *
	 * @param $item - row (key, value array)
	 * @return HTML
	 */
	function column_name($item) {

		$menu = Options_Framework_Admin::menu_settings();

		if ( isset( $menu['menu_custom_fields_slug'] ) ) {
			$menu_custom_fields_slug = $menu['menu_custom_fields_slug'];
		}

		// links going to /admin.php?page=[your_plugin_page][&other_params]
		// notice how we used $_REQUEST['page'], so action will be done on current page
		// also notice how we use $this->_args['singular'] so in this example it will
		// be something like &custom_field=2
		$actions = array(
			'edit' => sprintf('<a href="admin.php?page=%s&subpage=edit_form&slug=%s">%s</a>', esc_attr($menu_custom_fields_slug), esc_attr($item['slug']), esc_html__('Edit', 'datebook')),
		);

		// if foreign language do not exist, then show title in English
		$item['name'] = isset($item['name_' . DATEBOOK_WPLANG]) && !empty($item['name_' . DATEBOOK_WPLANG]) ? $item['name_' . DATEBOOK_WPLANG] : $item['name_en_US'];

		return sprintf('%s %s',
			esc_attr($item['name']),
			$this->row_actions($actions)
		);
	}


	/**
	 * [REQUIRED] This method return columns to display in table
	 * you can skip columns that you do not want to show
	 * like content, or description
	 *
	 * @return array
	 */
	function get_columns() {

		$columns = array(
			'name' => esc_html__('Title', 'datebook'),
			'slug' => esc_html__('Slug', 'datebook'),
			'field_items' => esc_html__('Field items', 'datebook'),
			'type' => esc_html__('Type', 'datebook'),
			'searchable' => esc_html__('Searchable', 'datebook'),
			'required' => esc_html__('Required', 'datebook'),
			'visible_to_admin_only' => esc_html__('Visible to admin', 'datebook'),
			'active' => esc_html__('Enable', 'datebook')
		);

		return $columns;

	}


	/**
	 * [REQUIRED] This is the most important method
	 *
	 * It will get rows from database and prepare them to be showed in table
	 */
	function prepare_items() {

		global $datebook_reserved_fields;

		$per_page = 1000; // constant, how much records will be shown per page

		$columns = $this->get_columns();
		$hidden = array();
		$sortable = $this->get_sortable_columns();

		// here we configure table headers, defined in our methods
		$this->_column_headers = array($columns, $hidden, $sortable);

		// [OPTIONAL] process bulk action if any
		$this->process_bulk_action();

		$get_profile_custom_fields = get_option('datebook_profile_custom_fields');

		foreach ($datebook_reserved_fields as $key => $value) {
			if(isset($get_profile_custom_fields[$key])){
				$datebook_reserved_fields[$key] = $get_profile_custom_fields[$key];
			}
		}
		
		$all_custom_fields = $datebook_reserved_fields;

		// Define $items array
		// notice that last argument is ARRAY_A, so we will retrieve array
		$this->items = $all_custom_fields;

		if($all_custom_fields && is_array($all_custom_fields)){
			// will be used in pagination settings
			$total_items = count($all_custom_fields);
		} else {
			$total_items = 0;
		}

		// prepare query params, as usual current page, order by and order direction
		$paged = isset($_REQUEST['paged']) ? ($per_page * max(0, intval($_REQUEST['paged']) - 1)) : 0;

		// configure pagination
		$this->set_pagination_args(array(
			'total_items' => $total_items, // total items defined above
			'per_page' => $per_page, // per page constant defined at top of method
			'total_pages' => ceil($total_items / $per_page) // calculate pages count
		));
	}
}


/**
 * DateBook_Custom_Fields_Table class that will display our custom table
 * records in nice table
 */
class DateBook_Custom_Fields_Table extends WP_List_Table {

	/**
	 * [REQUIRED] You must declare constructor and give some basic params
	 */
	function __construct() {

		global $status, $page;

		parent::__construct(array(
			'singular' => 'field',
			'plural' => 'fields',
		));
	}


	/**
	 * [REQUIRED] this is a default column renderer
	 *
	 * @param $item - row (key, value array)
	 * @param $column_name - string (key)
	 * @return HTML
	 */
	function column_default($item, $column_name) {
		
		if ($column_name == 'field_items'){
			
			// if foreign language do not exist, then show title in English
			$item[$column_name] = isset($item[$column_name . '_' . DATEBOOK_WPLANG]) ? $item[$column_name . '_' . DATEBOOK_WPLANG] : $item[$column_name . '_en_US'];

			return esc_attr($item[$column_name]);
			
		} else {
			
			return esc_attr($item[$column_name]);
			
		}
		
	}


	/**
	 * [OPTIONAL] this is example, how to render specific column
	 *
	 * method name must be like this: "column_[column_name]"
	 *
	 * @param $item - row (key, value array)
	 * @return HTML
	 */
	function column_searchable($item) {

		if (isset($item['searchable']) && $item['searchable'] == 1) {

			$out = '<span style="color:green">' . esc_html__('Yes', 'datebook') . '</span>';

		} else {

			$out = '';

		}

		return $out;

	}


	/**
	 * [OPTIONAL] this is example, how to render specific column
	 *
	 * method name must be like this: "column_[column_name]"
	 *
	 * @param $item - row (key, value array)
	 * @return HTML
	 */
	function column_required($item) {

		if (isset($item['required']) && $item['required'] == 1) {

			$out = '<span style="color:green">' . esc_html__('Yes', 'datebook') . '</span>';

		} else {

			$out = '';

		}

		return $out;

	}


	/**
	 * [OPTIONAL] this is example, how to render specific column
	 *
	 * method name must be like this: "column_[column_name]"
	 *
	 * @param $item - row (key, value array)
	 * @return HTML
	 */
	function column_visible_to_admin_only($item) {

		if (isset($item['visible_to_admin_only']) && $item['visible_to_admin_only'] == 1) {

			$out = '<span style="color:green">' . esc_html__('Yes', 'datebook') . '</span>';

		} else {

			$out = '';

		}

		return $out;

	}


	/**
	 * [OPTIONAL] this is example, how to render specific column
	 *
	 * method name must be like this: "column_[column_name]"
	 *
	 * @param $item - row (key, value array)
	 * @return HTML
	 */
	function column_active($item) {
		
		if (isset($item['active']) && $item['active'] == 1) {
			
			$out = '<span style="color:green">' . esc_html__('Yes', 'datebook') . '</span>';
			
		} else {
			
			$out = '';
			
		}

		return $out;

	}


	/**
	 * [OPTIONAL] this is example, how to render column with actions,
	 * when you hover row "Edit | Delete" links showed
	 *
	 * @param $item - row (key, value array)
	 * @return HTML
	 */
	function column_name($item) {

		$menu = Options_Framework_Admin::menu_settings();

		if ( isset( $menu['menu_custom_fields_slug'] ) ) {
			$menu_custom_fields_slug = $menu['menu_custom_fields_slug'];
		}

		// links going to /admin.php?page=[your_plugin_page][&other_params]
		// notice how we used $_REQUEST['page'], so action will be done on current page
		// also notice how we use $this->_args['singular'] so in this example it will
		// be something like &custom_field=2
		$actions = array(
			'edit' => sprintf('<a href="admin.php?page=%s&subpage=edit_form&slug=%s">%s</a>', esc_attr($menu_custom_fields_slug), esc_attr($item['slug']), esc_html__('Edit', 'datebook')),
			'delete' => sprintf('<a onclick="return confirm(\' %s \')" href="admin.php?page=%s&action=delete&slug=%s">%s</a>', esc_html__('Do you really want to delete it?', 'datebook'), esc_attr($menu_custom_fields_slug), esc_attr($item['slug']), esc_html__('Delete', 'datebook')),
		);

		// if foreign language do not exist, then show title in English
		$item['name'] = isset($item['name_' . DATEBOOK_WPLANG]) && !empty($item['name_' . DATEBOOK_WPLANG]) ? $item['name_' . DATEBOOK_WPLANG] : $item['name_en_US'];

		return sprintf('%s %s',
			esc_attr($item['name']),
			$this->row_actions($actions)
		);
	}


	/**
	 * [REQUIRED] This method return columns to display in table
	 * you can skip columns that you do not want to show
	 * like content, or description
	 *
	 * @return array
	 */
	function get_columns() {

		$columns = array(
			'name' => esc_html__('Title', 'datebook'),
			'slug' => esc_html__('Slug', 'datebook'),
			'field_items' => esc_html__('Field items', 'datebook'),
			'type' => esc_html__('Type', 'datebook'),
			'searchable' => esc_html__('Searchable', 'datebook'),
			'required' => esc_html__('Required', 'datebook'),
			'visible_to_admin_only' => esc_html__('Visible to admin', 'datebook'),
			'active' => esc_html__('Enable', 'datebook')
		);

		return $columns;

	}


	/**
	 * [OPTIONAL] This method return columns that may be used to sort table
	 * all strings in array - is column names
	 * notice that true on name column means that its default sort
	 *
	 * @return array
	 */
	function get_sortable_columns() {

		$sortable_columns = array(
			'name' => array('name', true),		
		);

		return $sortable_columns;

	}


	/**
	 * [OPTIONAL] This method processes bulk actions
	 * it can be outside of class
	 * it can not use wp_redirect coz there is output already
	 * in this example we are processing delete action
	 * message about successful deletion will be shown on page in next part
	 */
	function process_bulk_action() {

		if(isset($_REQUEST['action']) && $_REQUEST['action'] == 'delete'){

			$get_slug = isset($_REQUEST['slug']) ? $_REQUEST['slug'] : '';

			if (isset($get_slug)) {

				// Update entire array
				// Get entire array
				$get_profile_custom_fields = get_option('datebook_profile_custom_fields');

				$all_custom_fields = $get_profile_custom_fields;

				if ('delete' === $this->current_action()) {

					foreach ($all_custom_fields as $key => $value) {

						// Remove field from array
						if ($key == $get_slug) {
							unset($all_custom_fields[$key]);
						}

					}

					//Update entire array
					update_option('datebook_profile_custom_fields', $all_custom_fields);

				}
			}
		}
	}


	/**
	 * [REQUIRED] This is the most important method
	 *
	 * It will get rows from database and prepare them to be showed in table
	 */
	function custom_fields_items() {

		global $datebook_reserved_fields;

		$get_profile_custom_fields = get_option('datebook_profile_custom_fields');

		$all_custom_fields = array_merge($get_profile_custom_fields,$datebook_reserved_fields);

		return $all_custom_fields;

	}


	/**
	 * It will get rows from database and prepare them to be showed as select
	 */
	function get_all_profile_custom_fields($type_name, $custom_field_type) {

		// Define $items array
		// notice that last argument is ARRAY_A, so we will retrieve array
		$profile_custom_fields_array = array('title' => 'Title', 'text' => 'Text', 'textarea' => 'Textarea', 'radio' => 'Radio buttons', 'checkbox' => 'Check boxes (multi-select)', 'select' => 'Select (dropdown)', 'multiselect-single' => 'Select (single-select box)', 'multiselect' => 'Select (multi-select box)');

		$profile_custom_fields_out = '';
		$profile_custom_fields_out .= '<select name="' . $type_name . '">';
		$profile_custom_fields_out .= '<option value="0">' . esc_html__('Select field type', 'datebook') . '</option>';
		
		foreach($profile_custom_fields_array as $key => $value){

			if ($custom_field_type == $key) { $selected = ' selected="selected" '; } else { $selected = ''; }

			$profile_custom_fields_out .= '<option value="' . esc_attr($key) . '" '.$selected.'>' . esc_attr($value) . '</option>';
		}
		$profile_custom_fields_out .= '</select>';
		
		return $profile_custom_fields_out;

    }


	/**
	 * [REQUIRED] This is the most important method
	 *
	 * It will get rows from database and prepare them to be showed in table
	 */
	function prepare_items() {

		global $datebook_reserved_fields;

		$per_page = 1000; // constant, how much records will be shown per page

		$columns = $this->get_columns();
		$hidden = array();
		$sortable = $this->get_sortable_columns();

		// here we configure table headers, defined in our methods
		$this->_column_headers = array($columns, $hidden, $sortable);

		// [OPTIONAL] process bulk action if any
		$this->process_bulk_action();

		$get_profile_custom_fields = get_option('datebook_profile_custom_fields');

		foreach ($datebook_reserved_fields as $key => $value) {
			if(isset($datebook_reserved_fields[$key])){
				unset($get_profile_custom_fields[$key]);
			}
		}

		$all_custom_fields = $get_profile_custom_fields; //array_merge($get_profile_custom_fields,$datebook_reserved_fields);

		// Define $items array
		// notice that last argument is ARRAY_A, so we will retrieve array
		$this->items = $all_custom_fields;

		if(isset($all_custom_fields) && !empty($all_custom_fields) && is_array($all_custom_fields)){
			// will be used in pagination settings
			$total_items = count($all_custom_fields);
		} else {
			$total_items = 0;
		}

		// prepare query params, as usual current page, order by and order direction
		$paged = isset($_REQUEST['paged']) ? ($per_page * max(0, intval($_REQUEST['paged']) - 1)) : 0;

		// configure pagination
		$this->set_pagination_args(array(
			'total_items' => $total_items, // total items defined above
			'per_page' => $per_page, // per page constant defined at top of method
			'total_pages' => ceil($total_items / $per_page) // calculate pages count
		));
	}
}


class Options_Framework_Custom_Fields {

	/**
	 * List page handler
	 *
	 * This function renders our custom table
	 * Notice how we display message about successfull deletion
	 * Actualy this is very easy, and you can add as many features
	 * as you want.
	 *
	 * Look into /wp-admin/includes/class-wp-*-list-table.php for examples
	 */
	static function datebook_page_handler() {

		if (isset($_REQUEST['subpage']) && $_REQUEST['subpage'] == 'edit_form') {

			self::datebook_profile_custom_fields_form_page_handler();

		} else {

			$table_reserved = new DateBook_Reserved_Custom_Fields_Table();
			$table_reserved->prepare_items();

			$table = new DateBook_Custom_Fields_Table();
			$table->prepare_items();

			$message = '';
			if ($table->current_action() === 'delete') {
				if(is_array($_REQUEST['slug'])){
					$message = sprintf(esc_html__('Item deleted: %d', 'datebook'), count($_REQUEST['slug']));
				} else {
					$message = sprintf(esc_html__('Item deleted: %s', 'datebook'), $_REQUEST['slug']);
				}

			}

			$menu = Options_Framework_Admin::menu_settings();

			if ( isset( $menu['menu_custom_fields_slug'] ) ) {
				$menu_custom_fields_slug = $menu['menu_custom_fields_slug'];
			}

			?>
			
			<div class="wrap">

				<h1 class="wp-heading-inline"><?php esc_html_e('Custom Fields', 'datebook')?></h1>
				
					<a class="page-title-action" href="<?php echo get_admin_url(get_current_blog_id(), 'admin.php?page=' . $menu_custom_fields_slug . '&subpage=edit_form');?>"><?php esc_html_e('Add new', 'datebook')?></a>
			
					<hr class="wp-header-end">
			
					<div id="message" class="bg-white">
						<p><?php esc_html_e('On this page you can add, edit, delete custom fields.', 'datebook')?></p>
					</div>
			
					<?php 
				
					if (!empty($message)) {

						echo '<div id="message" class="updated below-h2"><p> ' . esc_attr($message) . '</p></div>';

					}

					?>
					
					<form id="profile_custom_fields-table" method="POST">
					
						<input type="hidden" name="page" value="<?php if ( isset($menu_custom_fields_slug) ) { echo esc_attr($menu_custom_fields_slug); } ?>"/>

						<table class="datebook-custom-table">
							<tr>
								<td><?php $table->display() ?></td>
							</tr>
						</table>

					</form>
					
					<h3><?php esc_html_e('Reserved fields', 'datebook')?></h3>
					<p><?php esc_html_e('These fields are reserved and have special functionality. They cannot be changed or replaced with other custom fields.', 'datebook')?></p>
					<table class="datebook-reserved-table">
						<tr>
							<td><?php $table_reserved->display() ?></td>
						</tr>
					</table>

				</div>

				<?php
			
		}
	}


	/**
	 * Simple function that validates data and retrieve bool on success
	 * and error message(s) on error
	 *
	 * @param $item
	 * @return bool|string
	 */
	static function datebook_validate_custom_field($item) {

		global $wp_locale_all;

		// get languages
		$languages_array = array();

		$languages_array['en_US'] = esc_html__( 'English', 'datebook' );

		$languages = DateBook_Localization::datebook_all_available_language_files();

		foreach($languages as $language) {
			if($language['checked'] == 1){
				$languages_array[$language['locale']] = $wp_locale_all[$language['locale']];
			}
		}

		$data = $update_data = array();
		$messages = array();

		if(is_array($item)) {

			if(empty($item['type'])){
				$messages[] = esc_html__('Custom Field type is required', 'datebook');
			}

			if(!empty($item['type']) && ($item['type'] == 'radio' || $item['type'] == 'checkbox' || $item['type'] == 'select') && empty($item['field_items'])){
				$messages[] = esc_html__('Field Items are required', 'datebook');
			}

			foreach ( $languages_array as $language_key => $language_name ) {

				if(!empty($item['type']) && $item['type'] == 'text' && !empty($item['name_' . $language_key])){

						$item['name_' . $language_key] = stripslashes(sanitize_textarea_field($item['name_' . $language_key]));

				} else {

					$messages[] = esc_html__('Title is required', 'datebook');

				}

				if(!empty($item['type']) && ($item['type'] == 'radio' || $item['type'] == 'checkbox' || $item['type'] == 'select') && !empty($item['field_items_' . $language_key])){

						$item['field_items_' . $language_key] = stripslashes(sanitize_textarea_field($item['field_items_' . $language_key]));

				} else {

					$messages[] = esc_html__('Field Items are required', 'datebook');

				}
			}


			$data['form_fields'] = $item;

			if (is_array($data)) {

				$data['success'] = true;

			} else {

				$data['success'] = false;
				$data['error_message'] = esc_html__( 'Data have not been saved', 'datebook' );

			}

		} else {

			$data['success'] = false;
			$data['error_message'] = esc_html__( 'Data have not been saved', 'datebook' );

		}

		return $data;

	}


	/**
	 * Form for adding andor editing row
	 * ============================================================================
	 *
	 * In this part you are going to add admin page for adding andor editing items
	 * You cant put all form into this function, but in this example form will
	 * be placed into meta box, and if you want you can split your form into
	 * as many meta boxes as you want
	 *
	 * http://codex.wordpress.org/Data_Validation
	 * http://codex.wordpress.org/Function_Reference/selected
	 */

	/**
	 * Form page handler checks is there some data posted and tries to save it
	 * Also it renders basic wrapper in which we are callin meta box render
	 */
	static function datebook_profile_custom_fields_form_page_handler() {

		$message = '';
		$notice = '';

		// here we are verifying does this request is post back and have correct nonce
		if (isset($_POST['nonce']) && wp_verify_nonce($_POST['nonce'], basename(__FILE__))) {

			// validate data, and if all ok save item to database
			// if id is zero insert otherwise update
			$item_data = stripslashes_deep(self::datebook_validate_custom_field($_POST));

			if ($item_data['success'] === true) {

				global $datebook_reserved_fields;

				$item = $item_data['form_fields'];

				if (empty($item['slug'])) {
					$item['slug'] = $item['name_en_US'];
					$replace_slug = 1;
				} else {
					$item['slug'] = $item['slug'];
					$replace_slug = 0;
				}

				// Let's lowercase, strip, clean, remove spaces
				$item['slug'] = DateBook_Utilities::datebook_makeslug($item['slug']);
				
				if($replace_slug == 1 && isset($datebook_reserved_fields[$item['slug']])){
					$item['slug'] = $item['slug'] . 'custom';
				}

				//Update entire array
				//Get entire array
				$get_profile_custom_fields = get_option('datebook_profile_custom_fields');

				//$all_custom_fields = array_merge($get_profile_custom_fields,$datebook_reserved_fields);
				$all_custom_fields = $get_profile_custom_fields;

				//Alter the options array appropriately
				$all_custom_fields[$item['slug']] = $item;
				
				//Update entire array
				$result = update_option('datebook_profile_custom_fields', $all_custom_fields);

				if( $result === FALSE){
					$notice = esc_html__('Data have NOT been saved', 'datebook');
				} else {
					$message = esc_html__('Item was successfully saved', 'datebook');
				}


			} else {

				// if $item_data not true it contains error message(s)
				$notice = $item_data['error_message'];

			}

		} else {

			// if this is not post back we load item to edit or give new one to create
			$item = '';

			if (isset($_REQUEST['slug'])) {

				global $datebook_reserved_fields;

				//Get entire array
				$get_profile_custom_fields = get_option('datebook_profile_custom_fields');

				//Alter the options array appropriately
				if(isset($get_profile_custom_fields[$_REQUEST['slug']])){
					$item = $get_profile_custom_fields[$_REQUEST['slug']];
				} else if(isset($datebook_reserved_fields[$_REQUEST['slug']])){
					$item = $datebook_reserved_fields[$_REQUEST['slug']];
				}

				if (!$item) {

					$item = $item_data['form_fields'];
					$notice = esc_html__('Item not found', 'datebook');

				}
			}
		}

		// here we adding our custom meta box
		add_meta_box('profile_custom_fields_form_meta_box', 'Edit Custom Field', array( __CLASS__ , 'datebook_profile_custom_fields_edit'), 'custom_field', 'normal', 'default');
	
		$menu = Options_Framework_Admin::menu_settings();

		$menu_custom_fields_slug = ( isset( $menu['menu_custom_fields_slug'] ) ) ? $menu['menu_custom_fields_slug'] : '';

    ?>

		<div class="wrap">

			<h1 class="wp-heading-inline"><?php esc_html_e('Edit Custom Field', 'datebook')?></h1>

			<a class="page-title-action" href="<?php echo get_admin_url(get_current_blog_id(), 'admin.php?page='.esc_attr($menu_custom_fields_slug));?>"><?php esc_html_e('Back to Custom Fields', 'datebook')?></a> 
			
			<a class="page-title-action" href="<?php echo get_admin_url(get_current_blog_id(), 'admin.php?page=' . $menu_custom_fields_slug . '&subpage=edit_form');?>"><?php esc_html_e('Add new', 'datebook')?></a>

			<hr class="wp-header-end">

			<?php if (!empty($notice)): ?>

				<div id="notice" class="error">
					<p><?php echo esc_attr($notice); ?></p>
				</div>

			<?php endif;?>

			<?php if (!empty($message)): ?>

				<div id="message" class="updated">
					<p><?php echo esc_attr($message) ?></p>
				</div>

			<?php endif;?>

			<form id="form" method="POST">

				<input type="hidden" name="nonce" value="<?php echo wp_create_nonce(basename(__FILE__))?>"/>
				
				<?php if (!empty($item['slug'])) { ?>
				
				<input type="hidden" name="slug" value="<?php echo esc_attr($item['slug']); ?>"/>
				
				<?php } ?>

				<div class="metabox-holder" id="poststuff">
					<div id="post-body">
						<div id="post-body-content">

							<?php /* And here we call our custom meta box */ ?>
							<?php do_meta_boxes('custom_field', 'normal', $item); ?>

							<input id="submit" class="button-primary" type="submit" name="submit" value="<?php esc_html_e('Save', 'datebook')?>" />

						</div>
					</div>
				</div>

			</form>

		</div>

	<?php

	}
	
	function shapeSpace_allowed_html() {

	$allowed_tags = array(
		'a' => array(
			'class' => array(),
			'href'  => array(),
			'rel'   => array(),
			'title' => array(),
		),
		'abbr' => array(
			'title' => array(),
		),
		'b' => array(),
		'blockquote' => array(
			'cite'  => array(),
		),
		'cite' => array(
			'title' => array(),
		),
		'code' => array(),
		'del' => array(
			'datetime' => array(),
			'title' => array(),
		),
		'dd' => array(),
		'div' => array(
			'class' => array(),
			'title' => array(),
			'style' => array(),
		),
		'dl' => array(),
		'dt' => array(),
		'em' => array(),
		'h1' => array(),
		'h2' => array(),
		'h3' => array(),
		'h4' => array(),
		'h5' => array(),
		'h6' => array(),
		'i' => array(),
		'img' => array(
			'alt'    => array(),
			'class'  => array(),
			'height' => array(),
			'src'    => array(),
			'width'  => array(),
		),
		'li' => array(
			'class' => array(),
		),
		'ol' => array(
			'class' => array(),
		),
		'p' => array(
			'class' => array(),
		),
		'q' => array(
			'cite' => array(),
			'title' => array(),
		),
		'span' => array(
			'class' => array(),
			'title' => array(),
			'style' => array(),
		),
		'strike' => array(),
		'strong' => array(),
		'ul' => array(
			'class' => array(),
		),
	);
	
	return $allowed_tags;
}

	/**
	 * This function renders our custom meta box
	 * $item is row
	 *
	 * @param $item
	 */
	static function datebook_profile_custom_fields_edit($item) {

		global $wp_locale_all, $datebook_reserved_fields;

		// get languages
		$languages_array = array();

		$languages_array['en_US'] = esc_html__( 'English', 'datebook' );

		$languages = DateBook_Localization::datebook_all_available_language_files();

		foreach($languages as $language) {
			if($language['checked'] == 1){
				$languages_array[$language['locale']] = $wp_locale_all[$language['locale']];
			}
		}


		if(isset($item['slug']) && isset($datebook_reserved_fields[$item['slug']]['slug']) && $item['slug'] == $datebook_reserved_fields[$item['slug']]['slug']){
			$enable = isset($datebook_reserved_fields[$item['slug']]['active_fields']['enable']) && $datebook_reserved_fields[$item['slug']]['active_fields']['enable'] == 1 ? 1 : 0;
			$type = isset($datebook_reserved_fields[$item['slug']]['active_fields']['type']) && $datebook_reserved_fields[$item['slug']]['active_fields']['type'] == 1 ? 1 : 0;
			$field_items = isset($datebook_reserved_fields[$item['slug']]['active_fields']['field_items']) && $datebook_reserved_fields[$item['slug']]['active_fields']['field_items'] == 1 ? 1 : 0;
			$characters_limit = isset($datebook_reserved_fields[$item['slug']]['active_fields']['characters_limit']) && $datebook_reserved_fields[$item['slug']]['active_fields']['characters_limit'] == 1 ? 1 : 0;
			$hidden_on_profile = isset($datebook_reserved_fields[$item['slug']]['active_fields']['hidden_on_profile']) && $datebook_reserved_fields[$item['slug']]['active_fields']['hidden_on_profile'] == 1 ? 1 : 0;
			$searchable = isset($datebook_reserved_fields[$item['slug']]['active_fields']['searchable']) && $datebook_reserved_fields[$item['slug']]['active_fields']['searchable'] == 1 ? 1 : 0;
			$required = isset($datebook_reserved_fields[$item['slug']]['active_fields']['required']) && $datebook_reserved_fields[$item['slug']]['active_fields']['required'] == 1 ? 1 : 0;
			$visible_to_admin_only = isset($datebook_reserved_fields[$item['slug']]['active_fields']['visible_to_admin_only']) && $datebook_reserved_fields[$item['slug']]['active_fields']['visible_to_admin_only'] == 1 ? 1 : 0;

			///////////////////////////////////////////
			/*if ($field_items == 1) {

				$get_terms_default_attributes = get_terms(array('taxonomy' => DATEBOOK_PROFILES_TAX, 'parent' => 0, 'hide_empty' => false));

				if ( !empty($get_terms_default_attributes) ) :
					foreach($languages_array as $language_key => $language_name){
						$output = '';
						foreach( $get_terms_default_attributes as $category ) {
							if( $category->parent == 0 ) {
								$output .= get_term_meta( $category->term_id, 'datebook_profiletype_' . $language_key, true ) . "\r\n";
							}
						}

						$item['field_items_' . $language_key] = $output;

					}
				endif;
			}*/
			//////////////////////////////////

		} else {
			$enable = 1;
			$type = 1;
			$field_items = 1;
			$characters_limit = 1;
			$hidden_on_profile = 1;
			$searchable = 1;
			$required = 1;
			$visible_to_admin_only = 1;
		}

		?>

		<table cellspacing="2" cellpadding="5" style="width: 100%;" class="form-table">
			<tbody>

				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="active"><?php esc_html_e('Enable', 'datebook'); ?></label>
					</th>
					<td>
					<?php if ($enable == 0) { ?>
						<input type="checkbox" checked="checked" disabled />
					<?php } ?>

					<span style="display:<?php if ($enable == 0) { echo 'none'; } ?>"><input id="datebook-active"  name="active" value="1" type="checkbox" <?php if (isset($item['active']) && $item['active'] == 1) { echo ' checked="checked"'; } ?> /></span> <label for="datebook-active"><?php esc_html_e('Yes', 'datebook'); ?></label></td>
				</tr>
				
				<tr class="form-field">
					<td valign="top" scope="row" colspan="2" bgcolor="#f7f9fb" style="padding-top:0; padding-bottom: 0;">
						<table width="100%" cellpadding="0" cellspacing="0" class="form-table">
							<thead style="border-bottom: 1px solid">
								<tr class="form-field">
									<th width="200px" valign="top" scope="row">
									</th>
									<td width="40%" valign="top" scope="row">
										<label><?php esc_html_e('Title displayed in the PROFILE form', 'datebook'); ?></label>
									</td>
									<td width="40%" valign="top" scope="row">
										<label><?php esc_html_e('Title displayed in the SEARCH form', 'datebook'); ?></label>
									</td>
								</tr>
							</thead>
							
							<?php foreach ( $languages_array as $language_key => $language_name ) { ?>
					
							<tr class="form-field">
								<th valign="top" scope="row">
									<label for="name"><?php echo sprintf( esc_html__( 'Title (%1$s)', 'datebook' ), $language_name ) ?></label>
								</th>
								<td><input id="name" name="name_<?php echo esc_attr($language_key); ?>" type="text" style="width: 95%" value="<?php if (isset($item['name_' . $language_key])) { echo esc_attr($item['name_' . $language_key]); } ?>" size="50" class="code" placeholder="<?php echo sprintf( esc_html__( 'Enter title in %1$s', 'datebook' ), esc_attr($language_name) ) ?>" /></td>
								<td><input id="name" name="name_search_<?php echo esc_attr($language_key); ?>" type="text" style="width: 95%" value="<?php if (isset($item['name_search_' . $language_key])) { echo esc_attr($item['name_search_' . $language_key]); } ?>" size="50" class="code" placeholder="<?php echo sprintf( esc_html__( 'Enter title in %1$s', 'datebook' ), esc_attr($language_name) ) ?>" /></td>
							</tr>
					
							<?php } ?>
							
							<tr class="form-field">
								<th valign="top" scope="row">
									
								</th>
								<td></td>
								<td style="padding-top:0"><em><?php esc_html_e('If titles do not differ in Profile and Search forms then leave fields empty in the Search form.', 'datebook'); ?></em></td>
							</tr>

						</table>
					</td>
				</tr>

				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="slug"><?php esc_html_e('Slug', 'datebook'); ?> (<?php esc_html_e('optional', 'datebook'); ?>)</label>
					</th>
					<td>
					
					<?php if (empty($item['slug'])) { ?>
					
						<input id="slug" name="slug" type="text" style="width: 95%" value="<?php if (isset($item['slug'])) { echo esc_attr($item['slug']); } ?>" size="50" class="code" placeholder="<?php esc_html_e('Slug', 'datebook'); ?>" />
						
					<?php } else {
					
						if (isset($item['slug'])) {
							
							echo '<strong>' . esc_attr($item['slug']) . '</strong><br />';
							
						}
					
					} ?>

					<em style="clear:both; width:100%; display:block"><?php esc_html_e('Slug is created only once and cannot be editted. It must be unique!', 'datebook'); ?></em>
						
						</td>
				</tr>
				
				
				<tr class="form-field" style="display:<?php if ($type == 0) { echo 'none'; } ?>">
					<td valign="top" bordercolor="#FFFFFF" scope="row" colspan="2" bgcolor="#f7f9fb" style="padding-top:0; padding-bottom: 0; border-bottom: 20px solid #fff;">
						<table width="100%" cellpadding="0" cellspacing="0" class="form-table">
							<thead style="border-bottom: 1px solid">
								<tr class="form-field">
									<th width="200px" valign="top" scope="row">
									</th>
									<td width="40%" valign="top" scope="row">
										<label><?php esc_html_e('Field type displayed in the PROFILE form', 'datebook'); ?></label>
									</td>
									<td width="40%" valign="top" scope="row">
										<label><?php esc_html_e('Field type displayed in the SEARCH form', 'datebook'); ?></label>
									</td>
								</tr>
							</thead>
							
							<tr class="form-field">
								<th valign="top" scope="row">
									<label for="name"><?php echo esc_html__( 'Field type', 'datebook' ); ?></label>
								</th>
								<td>

								<?php

								$item_type = (isset($item['type'])) ? $item['type'] : '';

								$all_custom_fields_types = new DateBook_Custom_Fields_Table;
								echo wp_specialchars_decode(esc_html($all_custom_fields_types->get_all_profile_custom_fields('type', $item_type)), ENT_QUOTES );

								?>

								</td>
								<td>

								<?php

								$item_type_search = (isset($item['type_search'])) ? $item['type_search'] : '';

								echo wp_specialchars_decode(esc_html($all_custom_fields_types->get_all_profile_custom_fields('type_search', $item_type_search)), ENT_QUOTES );

								?>
								</td>
							</tr>

						</table>
					</td>
				</tr>
				
				
				
				<tr class="form-field" style="display:<?php if ($field_items == 0) { echo 'none'; } ?>">
					<td valign="top" scope="row" colspan="2" bgcolor="#f7f9fb" style="padding-top:0; padding-bottom:0">
						<table width="100%" cellpadding="0" cellspacing="0" class="form-table">
							<thead style="border-bottom: 1px solid">
								<tr class="form-field">
									<th width="200px" valign="top" scope="row">
									</th>
									<td width="40%" valign="top" scope="row">
										<label><?php esc_html_e('Title displayed in the PROFILE form', 'datebook'); ?></label>
									</td>
									<td width="40%" valign="top" scope="row">
										<label><?php esc_html_e('Title displayed in the SEARCH form', 'datebook'); ?></label>
									</td>
								</tr>
							</thead>

							<?php foreach ( $languages_array as $language_key => $language_name ) { ?>

							<tr class="form-field">
								<th valign="top" scope="row">
									<label for="datebook-textarea"><?php echo sprintf( esc_html__( 'Field Items (%1$s)', 'datebook' ), esc_attr($language_name) ) ?></label>
								</th>
								<td style="padding-bottom:0">

									<textarea id="datebook-textarea" name="field_items_<?php echo esc_attr($language_key); ?>" rows="4"><?php if (!empty($item['field_items_' . $language_key])) { echo esc_attr($item['field_items_' . $language_key]); } ?></textarea>

								</td>
								<td style="padding-bottom:0">

									<textarea id="datebook-textarea" name="field_items_search_<?php echo esc_attr($language_key); ?>" rows="4"><?php if (!empty($item['field_items_search_' . $language_key])) { echo esc_attr($item['field_items_search_' . $language_key]); } ?></textarea>
									
								</td>
							</tr>
							
							<tr class="form-field">
								<th valign="top" scope="row">
									
								</th>
								<td colspan="2" style="padding-top:0"><em><?php esc_html_e('Enter items as many as you want. After each item press ENTER.', 'datebook'); ?></em></td>
							</tr>

							<?php } ?>
							
							<tr class="form-field">
								<th valign="top" scope="row">
									
								</th>
								<td></td>
								<td style="padding-top:0"><em><?php esc_html_e('If titles do not differ in Profile and Search forms then leave fields empty in the Search form.', 'datebook'); ?></em></td>
							</tr>

						</table>
					</td>
				</tr>


				<?php if (isset($item['slug']) && $item['slug'] == 'countrycity') { ?>

				<tr class="form-field">
					<td valign="top" bordercolor="#FFFFFF" scope="row" colspan="2" bgcolor="#f7f9fb" style="padding-top:0; padding-bottom: 0; border-bottom: 20px solid #fff;">
						<table width="100%" cellpadding="0" cellspacing="0" class="form-table">
							<thead style="border-bottom: 1px solid">
								<tr class="form-field">
									<th width="200px" valign="top" scope="row">
									</th>
									<td width="40%" valign="top" scope="row">
										<label><?php esc_html_e('Location displayed in the PROFILE form', 'datebook'); ?></label>
									</td>
									<td width="40%" valign="top" scope="row">
										<label><?php esc_html_e('Location displayed in the SEARCH form', 'datebook'); ?></label>
									</td>
								</tr>
							</thead>

							<tr class="form-field">
								<th valign="top" scope="row">
									<label for="name"><?php echo esc_html__( 'Location hierarchy', 'datebook' ); ?></label>
								</th>
								<td>

								<?php

								$countrycity_hierarchy = (isset($item['countrycity_hierarchy'])) ? $item['countrycity_hierarchy'] : 3;
								
								$datebook_countrycity_hierarchy = array(esc_html__( 'Country', 'datebook' ), esc_html__( 'Country', 'datebook' ) . ' -> ' . esc_html__( 'Region', 'datebook' ), esc_html__( 'Country', 'datebook' ) . ' -> ' . esc_html__( 'Region', 'datebook' ) . ' -> ' . esc_html__( 'City', 'datebook' ));

								echo '<select name="countrycity_hierarchy">';

								foreach ($datebook_countrycity_hierarchy as $key => $title) {
									
									$key = $key + 1;

									if ($key == $countrycity_hierarchy) { $selected = ' selected'; } else { $selected = ''; }

									echo '<option value="' . esc_attr($key) . '" ' . $selected . '>' . esc_attr($title) . '</option>';

								}

								echo '</select>';

								?>

								</td>
								<td>

								<?php

								$countrycity_hierarchy_search = (isset($item['countrycity_hierarchy_search'])) ? $item['countrycity_hierarchy_search'] : 3;

								echo '<select name="countrycity_hierarchy_search">';
								
								foreach ($datebook_countrycity_hierarchy as $key => $title) {

									$key = $key + 1;

									if ($key == $countrycity_hierarchy_search) { $selected = ' selected'; } else { $selected = ''; }

									echo '<option value="' . esc_attr($key) . '" ' . $selected . '>' . esc_attr($title) . '</option>';

								}

								echo '</select>';

								?>
								</td>
							</tr>

						</table>
					</td>
				</tr>

				<?php } ?>


				<tr class="form-field" style="display:<?php if ($characters_limit == 0) { echo 'none'; } ?>">
					<th valign="top" scope="row">
						<label for="datebook-limittext"><?php echo esc_html__( 'Characters limit', 'datebook' ); ?></label>
					</th>
					<td><input id="datebook-limittext" name="limittext" type="text" style="width: 150px" value="<?php if (isset($item['limittext'])) { echo esc_attr($item['limittext']); } ?>" size="5" maxlength="5" class="code" placeholder="<?php echo esc_html__( 'Enter number', 'datebook' ); ?>" /><br />

					<p><?php esc_html_e('How many characters allowed in text and textarea?', 'datebook'); ?></p>

					</td>
				</tr>


				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="active"><?php esc_html_e('Searchable', 'datebook'); ?></label>
					</th>
					<td><?php if ($searchable == 0) { ?>
						<input type="checkbox" checked="checked" disabled />
					<?php } ?>

					<span style="display:<?php if ($searchable == 0) { echo 'none'; } ?>"><input id="datebook-searchable"  name="searchable" value="1" type="checkbox" <?php if (isset($item['searchable']) && $item['searchable'] == 1) { echo ' checked="checked"'; } ?> /></span> <label for="datebook-searchable"><?php esc_html_e('Yes', 'datebook'); ?></label>
					
					<p><?php esc_html_e('It is displayed in the search form', 'datebook'); ?></p>
					</td>
				</tr>


				<?php if (isset($item['slug']) && ($item['slug'] != 'onlinestatus' && $item['slug'] != 'searchbyname' && $item['slug'] != 'searchbyid' && $item['slug'] != 'searchbyusername')) { ?>

				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="active"><?php esc_html_e('Required', 'datebook'); ?></label>
					</th>
					<td><?php if ($required == 0) { ?>
						<input type="checkbox" checked="checked" disabled />
					<?php } ?>

					<span style="display:<?php if ($required == 0) { echo 'none'; } ?>"><input id="datebook-required"  name="required" value="1" type="checkbox" <?php if (isset($item['required']) && $item['required'] == 1) { echo ' checked="checked"'; } ?> /></span> <label for="datebook-required"><?php esc_html_e('Yes', 'datebook'); ?></label>

					<p><?php esc_html_e('Users must fill in this field', 'datebook'); ?>. <?php esc_html_e('Important: if this field is required, then it must be used in at least one of the forms. Otherwise, the user will not be able to fill it out and his profile may be not visible to the public.', 'datebook'); ?></p>
					</td>
				</tr>

				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="active"><?php esc_html_e('Hidden on profile page', 'datebook'); ?></label>
					</th>
					<td><?php if ($hidden_on_profile == 0) { ?>
						<input type="checkbox" checked="checked" disabled />
					<?php } ?>

					<span style="display:<?php if ($hidden_on_profile == 0) { echo 'none'; } ?>"><input id="datebook-displayed-on-profile"  name="hidden_on_profile" value="1" type="checkbox" <?php if (isset($item['hidden_on_profile']) && $item['hidden_on_profile'] == 1) { echo ' checked="checked"'; } ?> /></span> <label for="datebook-displayed-on-profile"><?php esc_html_e('Yes', 'datebook'); ?></label>

					<p><?php esc_html_e('It is displayed on a profile page', 'datebook'); ?></p>
					</td>
				</tr>

				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="active"><?php esc_html_e('Visible to Administrator and author only', 'datebook'); ?></label>
					</th>
					<td><?php if ($visible_to_admin_only == 0) { ?>
						<input type="checkbox" checked="checked" disabled />
					<?php } ?>

					<span style="display:<?php if ($visible_to_admin_only == 0) { echo 'none'; } ?>"><input id="datebook-visible_to_admin_only"  name="visible_to_admin_only" value="1" type="checkbox" <?php if (isset($item['visible_to_admin_only']) && $item['visible_to_admin_only'] == 1) { echo ' checked="checked"'; } ?> /></span> <label for="datebook-visible_to_admin_only"><?php esc_html_e('Yes', 'datebook'); ?></label>

					<p><?php esc_html_e('This field will be still active but visible to administrator and owner of profile only', 'datebook'); ?></p>
					</td>
				</tr>
				
				<?php } else if (isset($item['slug']) && $item['slug'] == 'searchbyname') { ?>

				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="active"><?php esc_html_e('Search by Name', 'datebook'); ?></label>
					</th>
					<td>

					<input id="datebook-searchbyname"  name="searchbyname" value="1" type="checkbox" <?php if (isset($item['searchbyname']) && $item['searchbyname'] == 1) { echo ' checked="checked"'; } ?> /> <label for="datebook-searchbyname"><?php esc_html_e('Yes', 'datebook'); ?></label>
					
					</td>
				</tr>

				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="active"><?php esc_html_e('Search by ID', 'datebook'); ?></label>
					</th>
					<td>

					<input id="datebook-searchbyid"  name="searchbyid" value="1" type="checkbox" <?php if (isset($item['searchbyid']) && $item['searchbyid'] == 1) { echo ' checked="checked"'; } ?> /> <label for="datebook-searchbyid"><?php esc_html_e('Yes', 'datebook'); ?></label>
					
					</td>
				</tr>

				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="active"><?php esc_html_e('Search by username', 'datebook'); ?></label>
					</th>
					<td>

					<input id="datebook-searchbyusername"  name="searchbyusername" value="1" type="checkbox" <?php if (isset($item['searchbyusername']) && $item['searchbyusername'] == 1) { echo ' checked="checked"'; } ?> /> <label for="datebook-searchbyusername"><?php esc_html_e('Yes', 'datebook'); ?></label>
					
					</td>
				</tr>
				
				<?php } ?>

			</tbody>
		</table>
		
	<?php
	
	}

}