<?php
/**
 * @package   Options_Framework
 * @author    Devin Price <devin@wptheming.com>
 * @license   GPL-2.0+
 * @link      http://wptheming.com
 * @copyright 2010-2014 WP Theming
 */

class Options_Framework_Admin {

	/**
     * Page hook for the options screen
     *
     * @since 1.7.0
     * @type string
     */
	protected $options_screen = null;
	
	protected $options_settings_sub = null;
	protected $options_design_sub = null;
	protected $options_countries_sub = null;
	protected $options_subscription_plans_sub = null;
	protected $options_members_sub = null;
	protected $options_payments_sub = null;
	protected $options_custom_fields_sub = null;
	protected $options_profile_form_sub = null;
	protected $options_matching_sub = null;
	protected $options_gateways_sub = null;
	protected $options_theme_installer_sub = null;
	protected $options_countries_installer_sub = null;
	protected $options_notifications_sub = null;
	protected $options_shortcodes_sub = null;
	protected $options_migrate_sub = null;
	protected $options_referrer_sub = null;
	protected $options_copyright_sub = null;


    /**
     * Hook in the scripts and styles
     *
     * @since 1.7.0
     */
    public function init() {

		// Gets options to load
		$options = & Options_Framework::_optionsframework_options();
		
		// Gets designs to load
		$designs = & Options_Framework::_optionsframework_designs();
		
		// Checks if options are available
    	if ($options || $designs) {

			// Add the options page and menu item.
			add_action( 'admin_menu', array( $this, 'add_custom_options_page' ) );

			// Add the custom sub pages
			add_action( 'admin_menu', array( $this, 'add_custom_sub_pages' ) );
			
			// Add the required scripts and styles
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_styles' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );

			// Settings need to be registered after admin_init
			add_action( 'admin_init', array( $this, 'settings_init' ) );

			// Adds options menu to the admin bar
			add_action( 'wp_before_admin_bar_render', array( $this, 'optionsframework_admin_bar' ) );

		}

    }

	/**
     * Registers the settings
     *
     * @since 1.7.0
     */
    function settings_init() {

		// Get the option name
		$options_framework = new Options_Framework;

	    $name = $options_framework->get_option_name();

		// Registers the settings fields and callback
		register_setting( 'optionsframework', $name, array ( $this, 'validate_options' ) );

	    $name_design = $options_framework->get_design_name();

		// Registers the settings fields and callback
		register_setting( 'optionsframework_designs', $name_design, array ( $this, 'validate_design' ) );

		// Displays notice after options save
		add_action( 'optionsframework_save_css', array( $this, 'save_to_css_file' ) );

		// Displays notice after options save
		add_action( 'optionsframework_after_validate', array( $this, 'save_options_notice' ) );

		// Displays notice after designs save
		add_action( 'optionsframework_design_after_validate', array( $this, 'save_designs_notice' ) );

    }

	/*
	 * Define menu options
	 *
	 * Examples usage:
	 *
	 * add_filter( 'optionsframework_menu', function( $menu ) {
	 *     $menu['page_title'] = 'The Options';
	 *	   $menu['menu_title'] = 'The Options';
	 *     return $menu;
	 * });
	 *
	 * @since 1.7.0
	 *
	 */
	static function menu_settings() {

		$menu = array(

			// Modes: submenu, menu
            'mode' => 'submenu',

            // Submenu default settings
            'page_title' => esc_html__( 'DateBook Options', 'datebook' ),
			'menu_title' => esc_html__( 'DateBook', 'datebook' ),
			'capability' => 'edit_theme_options',
			'menu_slug' => 'datebook-options-framework',
            'parent_slug' => 'themes.php',

            // Menu default settings
            'icon_url' => 'dashicons-heart',

			'sub_page_settings_title' => esc_html__( 'Dashboard', 'datebook' ),
			'sub_menu_settings_title' => esc_html__( 'Dashboard', 'datebook' ),

			'sub_page_design_title' => esc_html__( 'Design and Style Options', 'datebook' ),
			'sub_menu_design_title' => esc_html__( 'Design and Style', 'datebook' ),
			'menu_design_slug' => 'datebook-options-design',

			'sub_page_countries_title' => esc_html__( 'Countries and Cities Options', 'datebook' ),
			'sub_menu_countries_title' => esc_html__( 'Countries and Cities', 'datebook' ),
			'menu_countries_slug' => 'datebook-options-countries',

			'sub_page_register_plans_title' => esc_html__( 'Subscriptions plans', 'datebook' ),
			'sub_menu_register_plans_title' => esc_html__( 'Subscription Plans', 'datebook' ),
			'menu_register_plans_slug' => 'datebook-options-subscription-plans',

			'sub_page_members_title' => esc_html__( 'Members', 'datebook' ),
			'sub_menu_members_title' => esc_html__( 'Members', 'datebook' ),
			'menu_members_slug' => 'datebook-options-members',

			'sub_page_payments_title' => esc_html__( 'Payments', 'datebook' ),
			'sub_menu_payments_title' => esc_html__( 'Payments', 'datebook' ),
			'menu_payments_slug' => 'datebook-options-payments',

			'sub_page_gateways_title' => esc_html__( 'Payment gateways', 'datebook' ),
			'sub_menu_gateways_title' => esc_html__( 'Gateways', 'datebook' ),
			'menu_gateways_slug' => 'datebook-options-gateways',

			'sub_page_custom_fields_title' => esc_html__( 'Custom Fields for Profile Options', 'datebook' ),
			'sub_menu_custom_fields_title' => esc_html__( 'Custom Fields', 'datebook' ),
			'menu_custom_fields_slug' => 'datebook-options-custom-fields',

			'sub_page_profile_form_title' => esc_html__( 'Form Options', 'datebook' ),
			'sub_menu_profile_form_title' => esc_html__( 'Forms', 'datebook' ),
			'menu_profile_form_slug' => 'datebook-options-forms',

			'sub_page_matching_title' => esc_html__( 'Matching Options', 'datebook' ),
			'sub_menu_matching_title' => esc_html__( 'Match Making', 'datebook' ),
			'menu_matching_slug' => 'datebook-options-matching',

			'sub_page_notifications_title' => esc_html__( 'Notification Options', 'datebook' ),
			'sub_menu_notifications_title' => esc_html__( 'Notifications', 'datebook' ),
			'menu_notifications_slug' => 'edit.php?post_type=' . DATEBOOK_NOTIFICATION_TYPE,

			'sub_page_cookies_title' => esc_html__( 'Cookies Options', 'datebook' ),
			'sub_menu_cookies_title' => esc_html__( 'Cookies', 'datebook' ),
			'menu_cookies_slug' => 'datebook-options-cookies',

			'sub_page_shortcodes_title' => esc_html__( 'Shortcodes', 'datebook' ),
			'sub_menu_shortcodes_title' => esc_html__( 'Shortcodes', 'datebook' ),
			'menu_shortcodes_slug' => 'datebook-shortcodes',

			'sub_page_referrer_title' => esc_html__( 'Referrers', 'datebook' ),
			'sub_menu_referrer_title' => esc_html__( 'Referrers', 'datebook' ),
			'menu_referrer_slug' => 'datebook-referrer',

			'sub_page_theme_installer_title' => esc_html__( 'Install Theme Data', 'datebook' ),
			'sub_menu_theme_installer_title' => esc_html__( 'Theme Installer', 'datebook' ),
			'menu_theme_installer_slug' => 'datebook-options-install',

			'sub_page_countries_installer_title' => esc_html__( 'Countries Installer', 'datebook' ),
			'sub_menu_countries_installer_title' => esc_html__( 'Countries Installer', 'datebook' ),
			'menu_countries_installer_slug' => 'datebook-countries-installer',

			'sub_page_demo_content_title' => esc_html__( 'Demo Content Installer', 'datebook' ),
			'sub_menu_demo_content_title' => esc_html__( 'Demo Installer', 'datebook' ),
			'menu_demo_content_slug' => 'datebook-demo-content',

			'sub_page_migrate_title' => esc_html__( 'Migrate', 'datebook' ),
			'sub_menu_migrate_title' => esc_html__( 'Migrate', 'datebook' ),
			'menu_migrate_slug' => 'datebook-migrate',

			'sub_page_copyright_title' => esc_html__( 'Copyright Options', 'datebook' ),
			'sub_menu_copyright_title' => esc_html__( 'License', 'datebook' ),
			'menu_copyright_slug' => 'datebook-options-copyright',

		);

		return apply_filters( 'optionsframework_menu', $menu );
	}

	/**
     * Add a subpage called "Plugin Options" to the appearance menu.
     *
     * @since 1.7.0
     */
	function add_custom_options_page() {

		$menu = $this->menu_settings();

		// If you want a top level menu, see this Gist:
		// https://gist.github.com/devinsays/884d6abe92857a329d99

		// Code removed because it conflicts with .org plugin check.

		$this->options_screen = add_menu_page(
            $menu['page_title'],
            $menu['menu_title'],
            $menu['capability'],
            $menu['menu_slug'],
            array( $this, 'options_page' ),
			$menu['icon_url'],
			50
        );

	}


	/**
     * Add coutries, custom fields, profile form, theme installer, countries installer and submenu item.
     *
     * @since 1.7.0
     */

	function add_custom_sub_pages() {

		$menu = $this->menu_settings();

		$this->options_settings_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_settings_title'], $menu['sub_menu_settings_title'], $menu['capability'], $menu['menu_slug'], array($this, 'options_page')
		);

		/*$this->options_design_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_design_title'], $menu['sub_menu_design_title'], $menu['capability'], $menu['menu_design_slug'], array($this, 'design_page')
		);*/

		$this->options_countries_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_countries_title'], $menu['sub_menu_countries_title'], $menu['capability'], $menu['menu_countries_slug'], array($this, 'countries_cities_page')
		);

		$this->options_subscription_plans_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_register_plans_title'], $menu['sub_menu_register_plans_title'], $menu['capability'], $menu['menu_register_plans_slug'], array($this, 'subscription_plans_page')
		);

		$this->options_members_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_members_title'], $menu['sub_menu_members_title'], $menu['capability'], $menu['menu_members_slug'], array($this, 'profile_members_page')
		);

		$this->options_payments_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_payments_title'], $menu['sub_menu_payments_title'], $menu['capability'], $menu['menu_payments_slug'], array($this, 'profile_payments_page')
		);

		$this->options_gateways_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_gateways_title'], $menu['sub_menu_gateways_title'], $menu['capability'], $menu['menu_gateways_slug'], array($this, 'gateways_page')
		);

		$this->options_custom_fields_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_custom_fields_title'], $menu['sub_menu_custom_fields_title'], $menu['capability'], $menu['menu_custom_fields_slug'], array($this, 'profile_custom_fields_page')
		);

		$this->options_profile_form_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_profile_form_title'], $menu['sub_menu_profile_form_title'], $menu['capability'], $menu['menu_profile_form_slug'], array($this, 'profile_form_page')
		);

		$this->options_matching_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_matching_title'], $menu['sub_menu_matching_title'], $menu['capability'], $menu['menu_matching_slug'], array($this, 'matching_page')
		);

		$this->options_notifications_sub = add_submenu_page($menu['menu_slug'], $menu['sub_page_notifications_title'], $menu['sub_menu_notifications_title'], $menu['capability'], $menu['menu_notifications_slug']);

		$this->options_cookies_sub = add_submenu_page($menu['menu_slug'], $menu['sub_page_cookies_title'], $menu['sub_menu_cookies_title'], $menu['capability'], $menu['menu_cookies_slug'], array($this, 'cookies_page')
		);

		$this->options_shortcodes_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_shortcodes_title'], $menu['sub_menu_shortcodes_title'], $menu['capability'], $menu['menu_shortcodes_slug'], array($this, 'shortcodes_page')
		);

		$this->options_referrer_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_referrer_title'], $menu['sub_menu_referrer_title'], $menu['capability'], $menu['menu_referrer_slug'], array($this, 'referrer_page')
		);

		$this->options_theme_installer_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_theme_installer_title'], $menu['sub_menu_theme_installer_title'], $menu['capability'], $menu['menu_theme_installer_slug'], array($this, 'theme_installer_page')
		);

		$this->options_countries_installer_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_countries_installer_title'], $menu['sub_menu_countries_installer_title'], $menu['capability'], $menu['menu_countries_installer_slug'], array($this, 'countries_installer_page')
		);

		$this->options_demo_content_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_demo_content_title'], $menu['sub_menu_demo_content_title'], $menu['capability'], $menu['menu_demo_content_slug'], array($this, 'demo_content_page')
		);

		$this->options_migrate_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_migrate_title'], $menu['sub_menu_migrate_title'], $menu['capability'], $menu['menu_migrate_slug'], array($this, 'migrate_page')
		);

		$this->options_copyright_sub = add_submenu_page( 
			$menu['menu_slug'], $menu['sub_page_copyright_title'], $menu['sub_menu_copyright_title'], $menu['capability'], $menu['menu_copyright_slug'], array($this, 'copyright_page')
		);

	}


	/**
     * Loads the required stylesheets
     *
     * @since 1.7.0
     */
	function enqueue_admin_styles( $hook ) {

		if ( $this->options_screen != $hook && $this->options_design_sub != $hook )
			return;

		wp_enqueue_style( 'optionsframework', OPTIONS_FRAMEWORK_DIRECTORY . 'css/optionsframework.css', array(),  Options_Framework::VERSION );
		wp_enqueue_style( 'wp-color-picker' );

	}

	/**
     * Loads the required javascript
     *
     * @since 1.7.0
     */
	function enqueue_admin_scripts( $hook ) {

		// Enqueue gradient color picker
		wp_enqueue_script('vanilla-picker', OPTIONS_FRAMEWORK_DIRECTORY . 'js/vanilla-picker.min.js', array(), Options_Framework::VERSION);
		
		wp_enqueue_script('popperjs', OPTIONS_FRAMEWORK_DIRECTORY . 'js/popper.min.js', array(), Options_Framework::VERSION);

		// Enqueue custom option panel JS
		wp_enqueue_script('options-custom', OPTIONS_FRAMEWORK_DIRECTORY . 'js/options-custom.js', array( 'jquery','wp-color-picker' ), Options_Framework::VERSION);

		wp_localize_script( 'options-custom', 'datebookdeletepopup', array(
			'datebook_delete_confirm_text' => esc_html__( 'Do you really want to delete it?', 'datebook' )
		));

		wp_enqueue_script( 'jquery-ui-droppable' );
		wp_enqueue_script( 'jquery-ui-sortable' );
		
		wp_enqueue_script('color-picker', OPTIONS_FRAMEWORK_DIRECTORY . 'js/color-gradient.js', array(), Options_Framework::VERSION, true);
		//wp_add_inline_script( 'vanilla-picker', '' );

		$screen = get_current_screen();

		if($screen->id =='profile' && $screen->post_type =='profile'){

			$gutenberg_active = 1;

			if (!function_exists('is_plugin_active')){
				include_once ABSPATH . 'wp-admin/includes/plugin.php';
			}
			if (is_plugin_active('classic-editor/classic-editor.php')){
				$editor_option = get_option( 'classic-editor-replace' );
				$block_editor_active = array( 'no-replace', 'block' );
				
				$gutenberg_active = 0;

				if(in_array( $editor_option, $block_editor_active)){
					$gutenberg_active = 0;
				}
			} else {
				// Gutenberg plugin is installed and activated.
				$gutenberg = !(false === has_filter('replace_editor', 'gutenberg_init'));

				// Block editor since 5.0.
				$block_editor = version_compare($GLOBALS['wp_version'], '5.0-beta', '>');

				if (!$gutenberg && !$block_editor){
					$gutenberg_active = 0;
				}
			}

			if($gutenberg_active == 1){

				$admin_profile_url = basename($_SERVER['REQUEST_URI']);

				if($admin_profile_url != 'profile.php'){

					wp_add_inline_script('jquery', 'jQuery(document).ready(function($) {"use strict"; (function(wp){ wp.data.dispatch( "core/notices" ).createNotice("warning", "' . esc_html__( 'Do not edit the profile data on this backend side. All profiles must be edited on the frontend.', 'datebook' ) . '", {isDismissible: false});} )( window.wp ); });');

				}
			}
		}

		
		// Inline scripts from options-interface.php
		add_action( 'admin_head', array( $this, 'of_admin_head' ) );
	}
	
	
	function of_admin_head() {
		// Hook to add custom scripts
		do_action( 'optionsframework_custom_scripts' );
	}


	/**
     * Builds out the options panel.
     *
	 * If we were using the Settings API as it was intended we would use
	 * do_settings_sections here.  But as we don't want the settings wrapped in a table,
	 * we'll call our own custom optionsframework_fields.  See options-interface.php
	 * for specifics on how each individual field is generated.
	 *
	 * Nonces are provided using the settings_fields()
	 *
     * @since 1.7.0
     */
	 function options_page() { ?>

		<div id="optionsframework-wrap" class="wrap">

		<?php $menu = $this->menu_settings(); ?>

		<h1><?php echo esc_html( $menu['page_title'] ); ?></h1>

		<hr class="wp-header-end">

		<?php settings_errors( 'datebook-options-framework' ); ?>

	    <div id="optionsframework-container">

		<ul class="datebook-framework-panel nav-tab-wrapper nav nav-tabs">
	        <?php echo Options_Framework_Interface::optionsframework_tabs('options'); ?>
	    </ul>

	    <div id="optionsframework-metabox" class="metabox-holder">
		    <div id="optionsframework" class="postbox">
				<form action="options.php" method="post">
				<input type="submit" class="button-primary float-right" style="margin-top:10px; margin-right: 10px" name="update" value="<?php esc_attr_e( 'Save', 'datebook' ); ?>" />
				<?php settings_fields( 'optionsframework' ); ?>
				<?php Options_Framework_Interface::optionsframework_fields('options'); /* Settings */ ?>
				<div id="optionsframework-submit">
					<input type="submit" class="button-primary" name="update" value="<?php esc_attr_e( 'Save', 'datebook' ); ?>" />
					<input type="submit" class="reset-button button-secondary" name="reset" value="<?php esc_attr_e( 'Restore Defaults', 'datebook' ); ?>" onclick="return confirm( '<?php print esc_js( esc_html__( 'Click OK to reset. Any plugin settings will be lost!', 'datebook' ) ); ?>' );" />
					<div class="clear"></div>
				</div>
				</form>
			</div> <!-- / #container -->
		</div>

		<?php do_action( 'optionsframework_after' ); ?>
		
		</div>
		
		</div> <!-- / .wrap -->

	<?php
	}


	/**
     * Builds out the Design panel.
     *
	 * If we were using the Settings API as it was intended we would use
	 * do_settings_sections here.  But as we don't want the settings wrapped in a table,
	 * we'll call our own custom optionsframework_fields.  See options-interface.php
	 * for specifics on how each individual field is generated.
	 *
	 * Nonces are provided using the settings_fields()
	 *
     * @since 2.0.0
     */
	 function design_page() { ?>

		<div id="optionsframework-wrap" class="wrap">

		<?php $menu = $this->menu_settings(); ?>

		<h1><?php echo esc_html( $menu['sub_page_design_title'] ); ?></h1>

		<hr class="wp-header-end">

	    <?php settings_errors( 'datebook-options-design' ); ?>

	    <div id="optionsframework-container">

		<ul class="datebook-design-panel nav-tab-wrapper nav nav-tabs">
	        <?php echo Options_Framework_Interface::optionsframework_tabs('design'); ?>
	    </ul>

	    <div id="optionsframework-metabox" class="metabox-holder">
		    <div id="optionsframework" class="postbox">
				<form action="options.php" method="post">
				<input type="submit" class="button-primary float-right" style="margin-top:10px; margin-right: 10px" name="update" value="<?php esc_attr_e( 'Save', 'datebook' ); ?>" />
				<?php settings_fields( 'optionsframework_designs' ); ?>
				<?php Options_Framework_Interface::optionsframework_fields('design'); /* Settings */ ?>
				<div id="optionsframework-submit">
					<input type="hidden" name="update_design" value="1" />
					<input type="submit" class="button-primary" name="update" value="<?php esc_attr_e( 'Save', 'datebook' ); ?>" />
					<input type="submit" class="reset-button button-secondary" name="reset_design" value="<?php esc_attr_e( 'Restore Defaults', 'datebook' ); ?>" onclick="return confirm( '<?php print esc_js( esc_html__( 'Click OK to reset. Any plugin settings will be lost!', 'datebook' ) ); ?>' );" />
					<div class="clear"></div>
				</div>
				</form>
			</div> <!-- / #container -->
		</div>

		<?php do_action( 'optionsframework_after' ); ?>
		
		</div>
		
		</div> <!-- / .wrap -->
		
	<?php
	}


	/**
     * Builds out the Countries and Cities panel.
     *
     * @since 1.7.0
     */
	 function countries_cities_page() {
		 
		echo Options_Framework_Countries::datebook_page_handler();
		
	}


	/**
     * Builds out the Registration plans (Subscriptions) panel.
     *
     * @since 2.1.0
     */
	 function subscription_plans_page() {
		 
		echo Options_Framework_Subscription_Plans::datebook_page_handler();
		
	}


	/**
     * Builds out the Members panel.
     *
     * @since 2.3
     */
	 function profile_members_page() {
		 
		echo DateBook_Submenu_Page_Members::datebook_page_handler();
		
	}


	/**
     * Builds out the Payments panel.
     *
     * @since 2.3
     */
	 function profile_payments_page() {
		 
		echo DateBook_Submenu_Page_Payments::datebook_page_handler();
		
	}


	/**
     * Builds out the Gateways panel.
     *
     * @since 3.4
     */
	 function gateways_page() {
		 
		echo DateBook_Payment_Gateways::gateways_page_handler();
		
	}


	/**
     * Builds out the Profile Custom Fields panel.
     *
     * @since 1.7.0
     */
	 function profile_custom_fields_page() {
		 
		echo Options_Framework_Custom_Fields::datebook_page_handler();
		
	}


	/**
     * Builds out the Profile Form panel.
     *
     * @since 1.7.0
     */
	 function profile_form_page() {
		 
		echo Options_Framework_Profile_Form::datebook_page_handler();
		
	}


	/**
     * Builds out the Match Making panel.
     *
     * @since 4.5.9
     */
	 function matching_page() {
		 
		echo DateBook_Matching_Options::datebook_interface();
		
	}


	/**
     * Builds out the Profile Form panel.
     *
     * @since 3.3
     */
	function cookies_page() {
		
		echo Class_DateBook_Cookies_Admin::cookies_page_handler();

	}


	/**
     * Builds out the Shortcodes panel.
     *
     * @since 3.4.0
     */
	 function shortcodes_page() {
		 
		echo Options_Framework_Shortcodes_List::datebook_register_shortcodes_handler();
		
	}


	/**
     * Builds out the Referrer panel.
     *
     * @since 4.4.9
     */
	 function referrer_page() {
		 
		echo Options_Framework_Referrer::register_handler();
		
	}


	/**
     * Builds out the Theme Installer panel.
     *
     * @since 1.7.0
     */
	 function theme_installer_page() {
		 
		echo DateBook_Install_Theme_Data::datebook_theme_installer_interface();
		
	}


	/**
     * Builds out the Countries Installer panel.
     *
     * @since 1.7.0
     */
	 function countries_installer_page() {
		 
		echo DateBook_Countries_Installer::datebook_interface();
		
	}


	/**
     * Builds out the Demo Content Installer panel.
     *
     * @since 4.4.9
     */
	 function demo_content_page() {
		 
		echo DateBook_Demo_Content_Installer::datebook_interface();
		
	}


	/**
     * Builds out the Migrate panel.
     *
     * @since 4.4
     */
	 function migrate_page() {
		 
		echo DateBook_Migrate::datebook_interface();
		
	}


	/**
     * Builds out the Copyright panel.
     *
     * @since 4.6.0
     */
	 function copyright_page() {

		 echo DateBook_Copyright::datebook_interface();

	}


	/**
	 * Validate Options.
	 *
	 * This runs after the submit/reset button has been clicked and
	 * validates the inputs.
	 *
	 * @uses $_POST['reset'] to restore default options
	 */
	function validate_options( $input ) {

		/*
		 * Restore Defaults.
		 *
		 * In the event that the user clicked the "Restore Defaults"
		 * button, the options defined in the plugin's options.php
		 * file will be added to the option for the active plugin.
		 */

		if ( isset( $_POST['reset'] ) ) {
			add_settings_error( 'datebook-options-framework', 'restore_defaults', esc_html__( 'Default options restored.', 'datebook' ), 'updated fade' );
			return $this->get_default_values('options');
		}


		/*
		 * Update Settings
		 *
		 * This used to check for $_POST['update'], but has been updated
		 * to be compatible with the theme customizer introduced in WordPress 3.4
		 */
		$css_type_array = array( 'checkbox', 'color', 'transparentcolor', 'gradientcolor', 'background', 'upload', 'images', 'typography', 'margin', 'padding', 'text', 'select' );

		$clean = $only_css = array();

		$options = & Options_Framework::_optionsframework_options();

		foreach ( $options as $option ) {

			if ( ! isset( $option['id'] ) ) {
				continue;
			}

			if ( ! isset( $option['type'] ) ) {
				continue;
			}

			$id = preg_replace( '/[^a-zA-Z0-9._\-]/', '', strtolower( $option['id'] ) );

			// Set checkbox to false if it wasn't sent in the $_POST
			if ( 'checkbox' == $option['type'] && ! isset( $input[$id] ) ) {
				$input[$id] = false;
			}

			// Set each item in the multicheck to false if it wasn't sent in the $_POST
			if ( isset($option['options']) && is_array($option['options']) && !empty($option['options']) && 'multicheck' == $option['type'] && ! isset( $input[$id] ) ) {
				foreach ( $option['options'] as $key => $value ) {
					$input[$id][$key] = false;
				}
			}

			if ( in_array($option['type'], $css_type_array) ) {
				if ( has_filter( 'of_sanitize_' . $option['type'] ) ) {
					$only_css[$id] = apply_filters( 'of_sanitize_' . $option['type'], $input[$id], $option );
				}
			}

			// For a value to be submitted to database it must pass through a sanitization filter
			if (isset($option['type']) &&  has_filter( 'of_sanitize_' . $option['type'] ) && isset($input[$id])) {

				if($option['type'] == 'textarea_emoji'){
					if (strpos($input[$id], '=') !== false) {
						$input[$id] = json_decode(base64_decode($input[$id]));
					}
				}

				$clean[$id] = apply_filters( 'of_sanitize_' . $option['type'], $input[$id], $option );

			}

		}

		// Save CSS custom style to CSS file
		do_action( 'optionsframework_save_css', $only_css );

		// Hook to run after validation
		do_action( 'optionsframework_after_validate', $clean );

		return $clean;
		
	}
	
	
	/**
	 * Lighten or Darken color
	 *
	 */
	function adjustBrightness($hexCode, $adjustPercent) {

		// #FFFFFF
		if ( strpos( $hexCode, '#' ) === 0 ) {

			preg_match('/^#?([0-9a-f]{2})([0-9a-f]{2})([0-9a-f]{2})$/i', $hexCode, $parts);

			$hexCode = ltrim($hexCode, '#');
			
			if (strlen($hexCode) == 3) {
				$hexCode = $hexCode[0] . $hexCode[0] . $hexCode[1] . $hexCode[1] . $hexCode[2] . $hexCode[2];
			}
			
			$hexCode = array_map('hexdec', str_split($hexCode, 2));
			foreach ($hexCode as & $color) {
				$adjustableLimit = $adjustPercent < 0 ? $color : 255 - $color;
				$adjustAmount = ceil($adjustableLimit * $adjustPercent);
				$color = str_pad(dechex($color + $adjustAmount), 2, '0', STR_PAD_LEFT);
			}
			
			return '#' . implode($hexCode);

		} else {

			// RGBA
			preg_match( '/^rgba?[\s+]?\([\s+]?(\d+)[\s+]?,[\s+]?(\d+)[\s+]?,[\s+]?(\d+)[\s+]?/i', $hexCode, $by_color );

	        $hexCode = sprintf( '#%02x%02x%02x', $by_color[1], $by_color[2], $by_color[3] );

			$hexCode = ltrim($hexCode, '#');

			if (strlen($hexCode) == 3) {
				$hexCode = $hexCode[0] . $hexCode[0] . $hexCode[1] . $hexCode[1] . $hexCode[2] . $hexCode[2];
			}
		
			$hexCode = array_map('hexdec', str_split($hexCode, 2));
		
			foreach ($hexCode as & $color) {
				$adjustableLimit = $adjustPercent < 0 ? $color : 255 - $color;
				$adjustAmount = ceil($adjustableLimit * $adjustPercent);
			
				$color = str_pad(dechex($color + $adjustAmount), 2, '0', STR_PAD_LEFT);
			}
		
			return '#' . implode($hexCode);
		
		}
	}


	/**
	 * Validate Design.
	 *
	 * This runs after the submit/reset button has been clicked and
	 * validates the inputs.
	 *
	 * @uses $_POST['reset'] to restore default options
	 */
	function validate_design( $input ) {

		/*
		 * Restore Defaults.
		 *
		 * In the event that the user clicked the "Restore Defaults"
		 * button, the options defined in the plugin's options.php
		 * file will be added to the option for the active plugin.
		 */

		if ( isset( $_POST['reset_design'] ) ) {
			add_settings_error( 'datebook-options-design', 'restore_defaults', esc_html__( 'Default options restored.', 'datebook' ), 'updated fade' );
			return $this->get_default_values('design');
		}

		/*
		 * Update Settings
		 *
		 * This used to check for $_POST['update'], but has been updated
		 * to be compatible with the theme customizer introduced in WordPress 3.4
		 */

		$css_type_array = array( 'checkbox', 'color', 'transparentcolor', 'gradientcolor', 'background', 'upload', 'images', 'typography', 'margin', 'padding', 'text', 'select' );

		$clean = $only_css = array();

		$options = & Options_Framework::_optionsframework_designs();

		foreach ( $options as $option ) {

			if ( ! isset( $option['id'] ) ) {
				continue;
			}

			if ( ! isset( $option['type'] ) ) {
				continue;
			}
			
			$id = preg_replace( '/[^a-zA-Z0-9._\-]/', '', strtolower( $option['id'] ) );

			// Set checkbox to false if it wasn't sent in the $_POST
			if ( 'checkbox' == $option['type'] && ! isset( $input[$id] ) ) {
				$input[$id] = false;
			}

			// Set each item in the multicheck to false if it wasn't sent in the $_POST
			if ( ('multicheck' == $option['type'] || 'multiupload' == $option['type']) && ! isset( $input[$id] ) ) {
				foreach ( $option['options'] as $key => $value ) {
					$input[$id][$key] = false;
				}
			}

			if ( in_array($option['type'], $css_type_array) ) {
				if ( has_filter( 'of_sanitize_' . $option['type'] ) ) {
					$only_css[$id] = apply_filters( 'of_sanitize_' . $option['type'], $input[$id], $option );
				}
			}

			// For a value to be submitted to database it must pass through a sanitization filter
			if (isset($option['type']) &&  has_filter( 'of_sanitize_' . $option['type'] ) && isset($input[$id])) {

				if($option['type'] == 'textarea_emoji'){
					if (strpos($input[$id], '=') !== false) {
						$input[$id] = json_decode(base64_decode($input[$id]));
					}
				}

				$clean[$id] = apply_filters( 'of_sanitize_' . $option['type'], $input[$id], $option );

			}
		}
		
		// Save CSS custom style to CSS file
		do_action( 'optionsframework_save_css', $only_css );

		// Hook to run after validation
		do_action( 'optionsframework_design_after_validate', $clean );

		return $clean;
	}


	/**
     * Shadow samples
     *
     * @since 4.4
     */
	 function shadows_list($shadow_selected_number) {

		$css = '';

		switch($shadow_selected_number) {
			case '1' : $css .= '-webkit-box-shadow: none !important; box-shadow: none !important;'; break;
			case '2' : $css .= 'box-shadow: 0 .125rem .25rem rgba(0,0,0,.075) !important;'; break;
			case '3' : $css .= 'box-shadow: 0 3px 6px 0 rgba(0,0,0,.12);'; break;			
			case '4' : $css .= 'box-shadow: rgba(206, 206, 220, 0.75) 0px 5px 11px;'; break;
			case '5' : $css .= 'box-shadow: 0 7px 30px -10px rgba(150,170,180,0.5);'; break;
			case '6' : $css .= 'box-shadow: 0  5px 10px rgba(154,160,185,0.05), 0 15px 40px rgba(166,173,201,0.2);'; break;
			case '7' : $css .= 'box-shadow: 0 .5rem 1rem rgba(0,0,0,.15)!important;'; break;
			case '8' : $css .= '-webkit-box-shadow:0 2px 5px 0 rgba(0,0,0,.16),0 2px 10px 0 rgba(0,0,0,.12)!important;box-shadow:0 2px 5px 0 rgba(0,0,0,.16),0 2px 10px 0 rgba(0,0,0,.12)!important;'; break;
			case '9' : $css .= '-webkit-box-shadow:0 5px 11px 0 rgba(0,0,0,.18),0 4px 15px 0 rgba(0,0,0,.15)!important;box-shadow:0 5px 11px 0 rgba(0,0,0,.18),0 4px 15px 0 rgba(0,0,0,.15)!important;'; break;
			case '10' : $css .= '-webkit-box-shadow:0 8px 17px 0 rgba(0,0,0,.2),0 6px 20px 0 rgba(0,0,0,.19)!important;box-shadow:0 8px 17px 0 rgba(0,0,0,.2),0 6px 20px 0 rgba(0,0,0,.19)!important;'; break;
			case '11' : $css .= 'box-shadow:0 1rem 3rem rgba(0,0,0,.175)!important;'; break;
			case '12' : $css .= '-webkit-box-shadow:0 12px 15px 0 rgba(0,0,0,.24),0 17px 50px 0 rgba(0,0,0,.19)!important;box-shadow:0 12px 15px 0 rgba(0,0,0,.24),0 17px 50px 0 rgba(0,0,0,.19)!important;'; break;
			case '13' : $css .= '-webkit-box-shadow:0 16px 28px 0 rgba(0,0,0,.22),0 25px 55px 0 rgba(0,0,0,.21)!important;box-shadow:0 16px 28px 0 rgba(0,0,0,.22),0 25px 55px 0 rgba(0,0,0,.21)!important;'; break;
			case '14' : $css .= '-webkit-box-shadow:0 27px 24px 0 rgba(0,0,0,.2),0 40px 77px 0 rgba(0,0,0,.22)!important;box-shadow:0 27px 24px 0 rgba(0,0,0,.2),0 40px 77px 0 rgba(0,0,0,.22)!important;'; break;
			default: $css .= ''; break;
		}

		return $css;

	}



	// Generate cache css
	function generate_css_data($css_data) {

		$recognized_fonts = of_recognized_font_faces();
		
		$css  = '';

		// Custom Typography
		if ($css_data['datebook_custom_typography']) {

			if( isset($css_data['typography_body']) ) {

				$css .= 'body {';

				if( !empty($css_data['typography_body']['size']) ) {
					$css .= 'font-size: ' . $css_data['typography_body']['size'] . $css_data['typography_body']['measure'] . ';';
				}
				if( !empty($css_data['typography_body']['face'])) {
					if ( array_key_exists( $css_data['typography_body']['face'], $recognized_fonts ) ) {
						$css .= 'font-family: ' . $recognized_fonts[$css_data['typography_body']['face']] . ';';
					}
				}
				if( !empty($css_data['typography_body']['style']) ) {
					$css .= 'font-style: ' . $css_data['typography_body']['style'] . ';';
				}
				if( !empty($css_data['typography_body']['color']) ) {
					$css .= 'color: ' . $css_data['typography_body']['color'] . ';';
				}
			
				$css .= '}';
			}

			if( !empty($css_data['link_color']) ) {
				$css .= 'a, .btn-link {color: ' . $css_data['link_color'] . '}';
			}

			if( !empty($css_data['typography_h1']) ) {
				$css .= 'h1, .h1 {';
				if( !empty($css_data['typography_h1']['size']) ) {
					$css .= 'font-size: ' . $css_data['typography_h1']['size'] . $css_data['typography_h1']['measure'] . ';';
				}
				if( !empty($css_data['typography_h1']['face']) ) {
					if ( array_key_exists( $css_data['typography_h1']['face'], $recognized_fonts ) ) {
						$css .= 'font-family: ' . $recognized_fonts[$css_data['typography_h1']['face']] . ';';
					}
				
				}
				if( !empty($css_data['typography_h1']['style']) ) {
					$css .= 'font-style: ' . $css_data['typography_h1']['style'] . ';';
				}
				if( !empty($css_data['typography_h1']['color']) ) {
					$css .= 'color: ' . $css_data['typography_h1']['color'] . ';';
				}
				$css .= '}';
			}

			if( !empty($css_data['typography_h2']) ) {
				$css .= 'h2, .h2 {';
				if( !empty($css_data['typography_h2']['size']) ) {
					$css .= 'font-size: ' . $css_data['typography_h2']['size'] . $css_data['typography_h2']['measure'] . ';';
				}
				if( !empty($css_data['typography_h2']['face']) ) {
					if ( array_key_exists( $css_data['typography_h2']['face'], $recognized_fonts ) ) {
						$css .= 'font-family: ' . $recognized_fonts[$css_data['typography_h2']['face']] . ';';
					}
				}
				if( !empty($css_data['typography_h2']['style']) ) {
					$css .= 'font-style: ' . $css_data['typography_h2']['style'] . ';';
				}
				if( !empty($css_data['typography_h2']['color']) ) {
					$css .= 'color: ' . $css_data['typography_h2']['color'] . ';';
				}
				$css .= '}';
			}

			if( !empty($css_data['typography_h3']) ) {
				$css .= 'h3, .h3 {';
				if( !empty($css_data['typography_h3']['size']) ) {
					$css .= 'font-size: ' . $css_data['typography_h3']['size'] . $css_data['typography_h3']['measure'] . ';';
				}
				if( !empty($css_data['typography_h3']['face']) ) {
					if ( array_key_exists( $css_data['typography_h3']['face'], $recognized_fonts ) ) {
						$css .= 'font-family: ' . $recognized_fonts[$css_data['typography_h3']['face']] . ';';
					}
				}
				if( !empty($css_data['typography_h3']['style']) ) {
					$css .= 'font-style: ' . $css_data['typography_h3']['style'] . ';';
				}
				if( !empty($css_data['typography_h3']['color']) ) {
					$css .= 'color: ' . $css_data['typography_h3']['color'] . ';';
				}
				$css .= '}';
			}

			if( !empty($css_data['typography_h4']) ) {
				$css .= 'h4, .h4 {';
				if( !empty($css_data['typography_h4']['size']) ) {
					$css .= 'font-size: ' . $css_data['typography_h4']['size'] . $css_data['typography_h4']['measure'] . ';';
				}
				if( !empty($css_data['typography_h4']['face']) ) {
					if ( array_key_exists( $css_data['typography_h4']['face'], $recognized_fonts ) ) {
						$css .= 'font-family: ' . $recognized_fonts[$css_data['typography_h4']['face']] . ';';
					}
				}
				if( !empty($css_data['typography_h4']['style']) ) {
					$css .= 'font-style: ' . $css_data['typography_h4']['style'] . ';';
				}
				if( !empty($css_data['typography_h4']['color']) ) {
					$css .= 'color: ' . $css_data['typography_h4']['color'] . ';';
				}
				$css .= '}';
			}

			if( !empty($css_data['typography_h5']) ) {
				$css .= 'h5, .h5 {';
				if( !empty($css_data['typography_h5']['size']) ) {
					$css .= 'font-size: ' . $css_data['typography_h5']['size'] . $css_data['typography_h5']['measure'] . ';';
				}
				if( !empty($css_data['typography_h5']['face']) ) {
					if ( array_key_exists( $css_data['typography_h5']['face'], $recognized_fonts ) ) {
						$css .= 'font-family: ' . $recognized_fonts[$css_data['typography_h5']['face']] . ';';
					}
				}
				if( !empty($css_data['typography_h5']['style']) ) {
					$css .= 'font-style: ' . $css_data['typography_h5']['style'] . ';';
				}
				if( !empty($css_data['typography_h5']['color']) ) {
					$css .= 'color: ' . $css_data['typography_h5']['color'] . ';';
				}
				$css .= '}';
			}

			if( !empty($css_data['typography_h6']) ) {
				$css .= 'h6, .h6 {';
				if( !empty($css_data['typography_h6']['size']) ) {
					$css .= 'font-size: ' . $css_data['typography_h6']['size'] . $css_data['typography_h6']['measure'] . ';';
				}
				if( !empty($css_data['typography_h6']['face']) ) {
					if ( array_key_exists( $css_data['typography_h6']['face'], $recognized_fonts ) ) {
						$css .= 'font-family: ' . $recognized_fonts[$css_data['typography_h6']['face']] . ';';
					}
				}
				if( !empty($css_data['typography_h6']['style']) ) {
					$css .= 'font-style: ' . $css_data['typography_h6']['style'] . ';';
				}
				if( !empty($css_data['typography_h6']['color']) ) {
					$css .= 'color: ' . $css_data['typography_h6']['color'] . ';';
				}
				$css .= '}';
			}

			if( !empty($css_data['typography_lead']) ) {
				$css .= '.lead {';
				if( !empty($css_data['typography_lead']['size']) ) {
					$css .= 'font-size: ' . $css_data['typography_lead']['size'] . $css_data['typography_lead']['measure'] . ';';
				}
				if( !empty($css_data['typography_lead']['face']) ) {
					if ( array_key_exists( $css_data['typography_lead']['face'], $recognized_fonts ) ) {
						$css .= 'font-family: ' . $recognized_fonts[$css_data['typography_lead']['face']] . ';';
					}
				}
				if( !empty($css_data['typography_lead']['style']) ) {
					$css .= 'font-style: ' . $css_data['typography_lead']['style'] . ';';
				}
				if( !empty($css_data['typography_lead']['color']) ) {
					$css .= 'color: ' . $css_data['typography_lead']['color'] . ';';
				}
				$css .= '}';
			}

			if( !empty($css_data['typography_blockquote']) ) {
				$css .= 'blockquote {';
				if( !empty($css_data['typography_blockquote']['size']) ) {
					$css .= 'font-size: ' . $css_data['typography_blockquote']['size'] . $css_data['typography_blockquote']['measure'] . ';';
				}
				if( !empty($css_data['typography_blockquote']['face']) ) {
					if ( array_key_exists( $css_data['typography_blockquote']['face'], $recognized_fonts ) ) {
						$css .= 'font-family: ' . $recognized_fonts[$css_data['typography_blockquote']['face']] . ';';
					}
				}
				if( !empty($css_data['typography_blockquote']['style']) ) {
					$css .= 'font-style: ' . $css_data['typography_blockquote']['style'] . ';';
				}
				if( !empty($css_data['typography_blockquote']['color']) ) {
					$css .= 'color: ' . $css_data['typography_blockquote']['color'] . ';';
				}
				$css .= '}';
			}


			if( !empty($css_data['text_grey']) ) {
				$css .= '.text-grey {color: ' . $css_data['text_grey'] . '}';
			}

			if( !empty($css_data['text_green']) ) {
				$css .= '.text-green {color: ' . $css_data['text_green'] . '}';
			}

			if( !empty($css_data['text_brown']) ) {
				$css .= '.text-brown {color: ' . $css_data['text_brown'] . '}';
			}

			if( !empty($css_data['text_yellow']) ) {
				$css .= '.amber-text {color: ' . $css_data['text_yellow'] . '}';
			}

			if( !empty($css_data['text_orange']) ) {
				$css .= '.text-orange {color: ' . $css_data['text_orange'] . '}';
			}

		}


		// Custom Styling
		if ($css_data['datebook_custom_styling']) {

			if( !empty($css_data['background_color']) ) {
				$css .= 'body {';
				$css .= 'background-color: ' . $css_data['background_color']['color'] . ';';
				$css .= 'background-image: url(' . $css_data['background_color']['image'] . ');';
				$css .= 'background-repeat: ' . $css_data['background_color']['repeat'] . ';';
				$css .= 'background-position: ' . $css_data['background_color']['position'] . ';';
				$css .= 'background-attachment: ' . $css_data['background_color']['attachment'] . ';';
				$css .= '}';
			}

			if( !empty($css_data['btn_shadow']) ) {

				$css .= '.btn {';

				$css .= $this->shadows_list($css_data['btn_shadow']);

				$css .= '}';

			}

			if( !empty($css_data['btn_default']) ) {
				$css .= '.btn-secondary, .label-default, .badge-default { background-color: ' . $css_data['btn_default'] . ';}';
			}

			if( !empty($css_data['btn_primary']) ) {
				$css .= '.btn-primary, .btn-outline-primary { background-color: ' . $css_data['btn_primary'] . ';}';
			}

			if( !empty($css_data['btn_message']) ) {
				$css .= '.btn-message { background-color: ' . $css_data['btn_message'] . ' !important;}';
			}

			if( !empty($css_data['fa_venus']) ) {
				$css .= 'i.fa-venus, .datebook-countries .datebook-subcountry .fa-venus {color: ' . $css_data['fa_venus'] . '}';
			}

			if( !empty($css_data['fa_mars']) ) {
				$css .= 'i.fa-mars, .datebook-countries .datebook-subcountry .fa-mars {color: ' . $css_data['fa_mars'] . '}';
			}

			if( !empty($css_data['popover_fa']) ) {
				$css .= '.popover i.fa-venus, .popover i.fa-mars {color: ' . $css_data['popover_fa'] . '}';
			}


			if( !empty($css_data['column_padding']) ) {
				$css .= '.datebook-padding-less {';
				if( !empty($css_data['column_padding']['padding_top']) ) {
					$css .= 'padding-top: ' . $css_data['column_padding']['padding_top'] . ';';
				}
				if( !empty($css_data['column_padding']['padding_right']) ) {
					$css .= 'padding-right: ' . $css_data['column_padding']['padding_right'] . ';';
				}
				if( !empty($css_data['column_padding']['padding_bottom']) ) {
				$css .= 'padding-bottom: ' . $css_data['column_padding']['padding_bottom'] . ';';
				}
				if( !empty($css_data['column_padding']['padding_left']) ) {
					$css .= 'padding-left: ' . $css_data['column_padding']['padding_left'] . ';';
				}
				$css .= '}';
			}

			if( !empty($css_data['typography_body']['color']) ) {
				$css .= '.datebook-front-title span {';
				$css .= 'border-bottom: 1px solid ' . $css_data['typography_body']['color'] . ';';
				$css .= '}';
			}
		
		}


		// Welcome Slogan
		if( !empty($css_data['welcome_slogan_font']) ) {
			$css .= '.datebook-welcome-panel h1 {';
			if( !empty($css_data['welcome_slogan_font']['size']) ) {
				$css .= 'font-size: ' . $css_data['welcome_slogan_font']['size'] . $css_data['welcome_slogan_font']['measure'] . ';';
			}
			if( !empty($css_data['welcome_slogan_font']['face']) ) {
				if ( array_key_exists( $css_data['welcome_slogan_font']['face'], $recognized_fonts ) ) {
					$css .= 'font-family: ' . $recognized_fonts[$css_data['welcome_slogan_font']['face']] . ';';
				}
			}
			if( !empty($css_data['welcome_slogan_font']['style']) ) {
				$css .= 'font-style: ' . $css_data['welcome_slogan_font']['style'] . ';';
			}
			if( !empty($css_data['welcome_slogan_font']['color']) ) {
				$css .= 'color: ' . $css_data['welcome_slogan_font']['color'] . ';';
			}
			$css .= '}';
		}
		if( !empty($css_data['welcome_slogan_align']) ) {

			if($css_data['welcome_slogan_align'] == 'right'){
				$welcome_slogan_align = 'right';
			} else if($css_data['welcome_slogan_align'] == 'center'){
				$welcome_slogan_align = 'center';
			} else if($css_data['welcome_slogan_align'] == 'left'){
				$welcome_slogan_align = 'left';
			} else {
				$welcome_slogan_align = '';
			}
			$css .= '.datebook-welcome-panel h1 {';
			$css .= 'text-align: ' . $welcome_slogan_align . ';';
			$css .= '}';
		}
		if( !empty($css_data['welcome_slogan_margin']) ) {
			$css .= '.datebook-welcome-panel h1 {';
			if( !empty($css_data['welcome_slogan_margin']['margin_top']) ) {
				$css .= 'margin-top: ' . $css_data['welcome_slogan_margin']['margin_top'] . ';';
			}
			if( !empty($css_data['welcome_slogan_margin']['margin_right']) ) {
				$css .= 'margin-right: ' . $css_data['welcome_slogan_margin']['margin_right'] . ';';
			}
			if( !empty($css_data['welcome_slogan_margin']['margin_bottom']) ) {
				$css .= 'margin-bottom: ' . $css_data['welcome_slogan_margin']['margin_bottom'] . ';';
			}
			if( !empty($css_data['welcome_slogan_margin']['margin_left']) ) {
				$css .= 'margin-left: ' . $css_data['welcome_slogan_margin']['margin_left'] . ';';
			}
			$css .= '}';
		}

		if( !empty($css_data['welcome_slogan_padding']) ) {
			$css .= '.datebook-welcome-panel h1 {';
			if( !empty($css_data['welcome_slogan_padding']['padding_top']) ) {
				$css .= 'padding-top: ' . $css_data['welcome_slogan_padding']['padding_top'] . ';';
			}
			if( !empty($css_data['welcome_slogan_padding']['padding_right']) ) {
				$css .= 'padding-right: ' . $css_data['welcome_slogan_padding']['padding_right'] . ';';
			}
			if( !empty($css_data['welcome_slogan_padding']['padding_bottom']) ) {
				$css .= 'padding-bottom: ' . $css_data['welcome_slogan_padding']['padding_bottom'] . ';';
			}
			if( !empty($css_data['welcome_slogan_padding']['padding_left']) ) {
				$css .= 'padding-left: ' . $css_data['welcome_slogan_padding']['padding_left'] . ';';
			}
			$css .= '}';
		}

		// Logo align
		/*if( !empty($css_data['website_logo_front']) ) {
			$css .= '.datebook-logo {';
			$css .= 'background-image: url(' . $css_data['website_logo_front'] . ');';
			$css .= '}';
		}*/

		// Logo background color - Front
		if( !empty($css_data['logo_bg_color_front']) ) {
			$css .= '.datebook-front-container .datebook-logo-top, .datebook-front-container .datebook-logo-inheader {';
			$css .= 'background-color: ' . $css_data['logo_bg_color_front'] . ';';
			$css .= '}';
		}
		// Logo background color - Other
		if( !empty($css_data['logo_bg_color_other']) ) {
			$css .= '.datebook-other-container .datebook-logo-top, .datebook-other-container .datebook-logo-inheader {';
			$css .= 'background-color: ' . $css_data['logo_bg_color_other'] . ';';
			$css .= '}';
		}


		// Header Image and Background color
		if( !empty($css_data['header_gradientcolor_front']) || !empty($css_data['header_bg_image_front']) ) {
			$css .= '.datebook-front-container #datebook-header {';
			$css .= 'background: #000 url(' . $css_data['header_bg_image_front']['image'] . ') no-repeat center center;';
			$css .= 'background: -webkit-gradient(linear, left top, left bottom, color-stop(0, rgba(0, 0, 0, 0.9)), color-stop(100, rgba(0, 0, 0, 0.4))) repeat scroll 0 0, rgba(0, 0, 0, 0) url(' . $css_data['header_bg_image_front']['image'] . ') ' . $css_data['header_bg_image_front']['repeat'] . ' ' . $css_data['header_bg_image_front']['attachment'] . ' ' . $css_data['header_bg_image_front']['position'] . ' / cover;';
			$css .= 'background: -ms-' . $css_data['header_gradientcolor_front'] . ' repeat scroll 0 0, rgba(0, 0, 0, 0) url(' . $css_data['header_bg_image_front']['image'] . ') ' . $css_data['header_bg_image_front']['repeat'] . ' ' . $css_data['header_bg_image_front']['attachment'] . ' ' . $css_data['header_bg_image_front']['position'] . ' / cover;';
			$css .= 'background: -moz-' . $css_data['header_gradientcolor_front'] . ' repeat scroll 0 0, rgba(0, 0, 0, 0) url(' . $css_data['header_bg_image_front']['image'] . ') ' . $css_data['header_bg_image_front']['repeat'] . ' ' . $css_data['header_bg_image_front']['attachment'] . ' ' . $css_data['header_bg_image_front']['position'] . ' / cover;';
			$css .= 'background: -o-' . $css_data['header_gradientcolor_front'] . ' repeat scroll 0 0, rgba(0, 0, 0, 0) url(' . $css_data['header_bg_image_front']['image'] . ') ' . $css_data['header_bg_image_front']['repeat'] . ' ' . $css_data['header_bg_image_front']['attachment'] . ' ' . $css_data['header_bg_image_front']['position'] . ' / cover;';
			$css .= 'background: -webkit-' . $css_data['header_gradientcolor_front'] . ' repeat scroll 0 0, rgba(0, 0, 0, 0) url(' . $css_data['header_bg_image_front']['image'] . ') ' . $css_data['header_bg_image_front']['repeat'] . ' ' . $css_data['header_bg_image_front']['attachment'] . ' ' . $css_data['header_bg_image_front']['position'] . ' / cover;';
			$css .= 'background: ' . $css_data['header_gradientcolor_front'] . ' repeat scroll 0 0, rgba(0, 0, 0, 0) url(' . $css_data['header_bg_image_front']['image'] . ') ' . $css_data['header_bg_image_front']['repeat'] . ' ' . $css_data['header_bg_image_front']['attachment'] . ' ' . $css_data['header_bg_image_front']['position'] . ' / cover;';
			$css .= '}';
		}

		if( !empty($css_data['header_gradientcolor_other']) || !empty($css_data['header_bg_image_other']) ) {
			$css .= '.datebook-other-container #datebook-header {';
			$css .= 'background: #000 url(' . $css_data['header_bg_image_other']['image'] . ') no-repeat center center;';
			$css .= 'background: -webkit-gradient(linear, left top, left bottom, color-stop(0, rgba(0, 0, 0, 0.9)), color-stop(100, rgba(0, 0, 0, 0.4))) repeat scroll 0 0, rgba(0, 0, 0, 0) url(' . $css_data['header_bg_image_other']['image'] . ') ' . $css_data['header_bg_image_other']['repeat'] . ' ' . $css_data['header_bg_image_other']['attachment'] . ' ' . $css_data['header_bg_image_other']['position'] . ' / cover;';
			$css .= 'background: -ms-' . $css_data['header_gradientcolor_other'] . ' repeat scroll 0 0, rgba(0, 0, 0, 0) url(' . $css_data['header_bg_image_other']['image'] . ') ' . $css_data['header_bg_image_other']['repeat'] . ' ' . $css_data['header_bg_image_other']['attachment'] . ' ' . $css_data['header_bg_image_other']['position'] . ' / cover;';
			$css .= 'background: -moz-' . $css_data['header_gradientcolor_other'] . ' repeat scroll 0 0, rgba(0, 0, 0, 0) url(' . $css_data['header_bg_image_other']['image'] . ') ' . $css_data['header_bg_image_other']['repeat'] . ' ' . $css_data['header_bg_image_other']['attachment'] . ' ' . $css_data['header_bg_image_other']['position'] . ' / cover;';
			$css .= 'background: -o-' . $css_data['header_gradientcolor_other'] . ' repeat scroll 0 0, rgba(0, 0, 0, 0) url(' . $css_data['header_bg_image_other']['image'] . ') ' . $css_data['header_bg_image_other']['repeat'] . ' ' . $css_data['header_bg_image_other']['attachment'] . ' ' . $css_data['header_bg_image_other']['position'] . ' / cover;';
			$css .= 'background: -webkit-' . $css_data['header_gradientcolor_other'] . ' repeat scroll 0 0, rgba(0, 0, 0, 0) url(' . $css_data['header_bg_image_other']['image'] . ') ' . $css_data['header_bg_image_other']['repeat'] . ' ' . $css_data['header_bg_image_other']['attachment'] . ' ' . $css_data['header_bg_image_other']['position'] . ' / cover;';
			$css .= 'background: ' . $css_data['header_gradientcolor_other'] . ' repeat scroll 0 0, rgba(0, 0, 0, 0) url(' . $css_data['header_bg_image_other']['image'] . ') ' . $css_data['header_bg_image_other']['repeat'] . ' ' . $css_data['header_bg_image_other']['attachment'] . ' ' . $css_data['header_bg_image_other']['position'] . ' / cover;';
			$css .= '}';
		}

		// Header Image and Background color
		if( !empty($css_data['welcome_bg_image']) ) {
			$css .= '.datebook-front-container .datebook-welcome-image {';
			$css .= 'background: transparent url(' . $css_data['welcome_bg_image']['image'] . ') ' . $css_data['welcome_bg_image']['repeat'] . ' ' . $css_data['welcome_bg_image']['position'] . ';';
			$css .= '}';
		}


		// Top menu
		if( !empty($css_data['top_menu_font_front']) ) {
			$css .= '.datebook-front-container #datebook-header .datebook-top-menu a, .datebook-front-container #datebook-header .datebook-top-menu .nav-link {';
			if( !empty($css_data['top_menu_font_front']['size']) ) {
				$css .= 'font-size: ' . $css_data['top_menu_font_front']['size'] . $css_data['top_menu_font_front']['measure'] . ';';
			}
			if( !empty($css_data['top_menu_font_front']['face']) ) {
				if ( array_key_exists( $css_data['top_menu_font_front']['face'], $recognized_fonts ) ) {
					$css .= 'font-family: ' . $recognized_fonts[$css_data['top_menu_font_front']['face']] . ';';
				}
			}
			if( !empty($css_data['top_menu_font_front']['style']) ) {
				$css .= 'font-style: ' . $css_data['top_menu_font_front']['style'] . ';';
			}
			if( !empty($css_data['top_menu_font_front']['color']) ) {
				$css .= 'color: ' . $css_data['top_menu_font_front']['color'] . ';';
			}
			$css .= '}';
		}
		
		if( !empty($css_data['top_menu_font_other']) ) {
			$css .= '.datebook-other-container #datebook-header .datebook-top-menu a, .datebook-other-container #datebook-header .datebook-top-menu .nav-link {';
			if( !empty($css_data['top_menu_font_other']['size']) ) {
				$css .= 'font-size: ' . $css_data['top_menu_font_other']['size'] . $css_data['top_menu_font_other']['measure'] . ';';
			}
			if( !empty($css_data['top_menu_font_other']['face']) ) {
				if ( array_key_exists( $css_data['top_menu_font_other']['face'], $recognized_fonts ) ) {
					$css .= 'font-family: ' . $recognized_fonts[$css_data['top_menu_font_other']['face']] . ';';
				}
			}
			if( !empty($css_data['top_menu_font_other']['style']) ) {
				$css .= 'font-style: ' . $css_data['top_menu_font_other']['style'] . ';';
			}
			if( !empty($css_data['top_menu_font_other']['color']) ) {
				$css .= 'color: ' . $css_data['top_menu_font_other']['color'] . ';';
			}
			$css .= '}';
		}

		// Top menu background color
		if( !empty($css_data['top_menu_bg_color_front']) ) {
			$css .= '.datebook-front-container .datebook-top-menu, .datebook-front-container .datebook-top-menu .dropdown-menu {';
			$css .= 'background-color: ' . $css_data['top_menu_bg_color_front'] . ' !important;';
			$css .= '}';

			$css .= '.datebook-front-container .datebook-top-menu .navbar-toggler {';
			$css .= 'background-color: ' . $css_data['top_menu_toggler_bg_color_front'] . ';';
			$css .= '}';
		}
		if( !empty($css_data['top_menu_bg_color_other']) ) {
			$css .= '.datebook-other-container .datebook-top-menu, .datebook-other-container .datebook-top-menu .dropdown-menu {';
			$css .= 'background-color: ' . $css_data['top_menu_bg_color_other'] . ' !important;';
			$css .= '}';

			$css .= '.datebook-other-container .datebook-top-menu .navbar-toggler {';
			$css .= 'background-color: ' . $css_data['top_menu_toggler_bg_color_other'] . ';';
			$css .= '}';
		}

		// Top menu margin
		if( !empty($css_data['top_menu_margin_front']) ) {
			$css .= '.datebook-front-container .datebook-top-menu {';
			if( !empty($css_data['top_menu_margin_front']['margin_top']) ) {
				$css .= 'margin-top: ' . $css_data['top_menu_margin_front']['margin_top'] . ';';
			}
			if( !empty($css_data['top_menu_margin_front']['margin_right']) ) {
				$css .= 'margin-right: ' . $css_data['top_menu_margin_front']['margin_right'] . ';';
			}
			if( !empty($css_data['top_menu_margin_front']['margin_bottom']) ) {
				$css .= 'margin-bottom: ' . $css_data['top_menu_margin_front']['margin_bottom'] . ';';
			}
			if( !empty($css_data['top_menu_margin_front']['margin_left']) ) {
				$css .= 'margin-left: ' . $css_data['top_menu_margin_front']['margin_left'] . ';';
			}
			$css .= '}';
		}
		if( !empty($css_data['top_menu_padding_front']) ) {
			$css .= '.datebook-front-container .datebook-top-menu {';
			if( !empty($css_data['top_menu_padding_front']['padding_top']) ) {
				$css .= 'padding-top: ' . $css_data['top_menu_padding_front']['padding_top'] . ';';
			}
			if( !empty($css_data['top_menu_padding_front']['padding_right']) ) {
				$css .= 'padding-right: ' . $css_data['top_menu_padding_front']['padding_right'] . ';';
			}
			if( !empty($css_data['top_menu_padding_front']['padding_bottom']) ) {
				$css .= 'padding-bottom: ' . $css_data['top_menu_padding_front']['padding_bottom'] . ';';
			}
			if( !empty($css_data['top_menu_padding_front']['padding_left']) ) {
				$css .= 'padding-left: ' . $css_data['top_menu_padding_front']['padding_left'] . ';';
			}
			$css .= '}';
		}


		// Logo align on Home page
		if( !empty($css_data['logo_align_front']) && $css_data['logo_align_front'] == 'left' ) {
			$css .= '.datebook-front-container .datebook-logo {';
			$css .= 'margin-right: auto;';
			$css .= '}';
		} else if( !empty($css_data['logo_align_front']) && $css_data['logo_align_front'] == 'center' ) {
			$css .= '.datebook-front-container .datebook-logo {';
			$css .= 'margin-right: auto;';
			$css .= 'margin-left: auto;';
			$css .= '}';
		} else if( !empty($css_data['logo_align_front']) && $css_data['logo_align_front'] == 'right' ) {
			$css .= '.datebook-front-container .datebook-logo {';
			$css .= 'margin-left: auto;';
			$css .= '}';
		}
		// Logo align on Other page
		if( !empty($css_data['logo_align_other']) && $css_data['logo_align_other'] == 'left' ) {
			$css .= '.datebook-other-container .datebook-logo {';
			$css .= 'margin-right: auto;';
			$css .= '}';
		} else if( !empty($css_data['logo_align_other']) && $css_data['logo_align_other'] == 'center' ) {
			$css .= '.datebook-other-container .datebook-logo {';
			$css .= 'margin-right: auto;';
			$css .= 'margin-left: auto;';
			$css .= '}';
		} else if( !empty($css_data['logo_align_front']) && $css_data['logo_align_other'] == 'right' ) {
			$css .= '.datebook-other-container .datebook-logo {';
			$css .= 'margin-left: auto;';
			$css .= '}';
		}
		
		// Logo margin right
		if((!empty($css_data['logo_position_front']) && $css_data['logo_position_front'] == 'insidemenu' && !empty($css_data['primary_menu_width_front']) && $css_data['primary_menu_width_front'] == 'full') || (!empty($css_data['logo_position_other']) && $css_data['logo_position_other'] == 'insidemenu' && !empty($css_data['primary_menu_width_other']) && $css_data['primary_menu_width_other'] == 'full')) {
			$css .= '.datebook-front-container .datebook-logo, .datebook-other-container .datebook-logo {';
			$css .= 'margin-right: 1rem;';
			$css .= '}';
		}


		// Primary menu subcontainer width on Home page
		if(!empty($css_data['primary_menu_width_front']) && $css_data['primary_menu_width_front'] == 'boxed') {
			$css .= '.datebook-front-container .datebook-logo {';
			$css .= 'margin-right: 1rem;';
			$css .= '}';

			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer {';
			$css .= 'width: auto;';
			$css .= '}';
		}
		if(!empty($css_data['primary_menu_width_other']) && $css_data['primary_menu_width_other'] == 'boxed') {
			$css .= '.datebook-other-container .datebook-logo {';
			$css .= 'margin-right: 1rem;';
			$css .= '}';

			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer {';
			$css .= 'width: auto;';
			$css .= '}';
		}


		// Primary menu class align on Home page
		if(!empty($css_data['primary_menu_align_front']) && $css_data['primary_menu_align_front'] == 'left') {
			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu-class {';
			$css .= 'margin-right: auto;';
			$css .= '}';
			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu {';
			$css .= 'margin-right: auto;';
			$css .= '}';
		} else if(!empty($css_data['primary_menu_align_front']) && $css_data['primary_menu_align_front'] == 'center') {
			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu-class {';
			$css .= 'margin-right: auto;';
			$css .= 'margin-left: auto;';
			$css .= 'max-width: max-content;';
			$css .= '}';
			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu {';
			$css .= 'margin-right: auto;';
			$css .= 'margin-left: auto;';
			$css .= '}';
		} else if(!empty($css_data['primary_menu_align_front']) && $css_data['primary_menu_align_front'] == 'right') {
			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu-class {';
			$css .= 'margin-left: auto;';
			$css .= '}';
			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu {';
			$css .= 'margin-left: auto;';
			$css .= '}';
		}
		// Primary menu align on Other page
		if(!empty($css_data['primary_menu_align_other']) && $css_data['primary_menu_align_other'] == 'left') {
			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu-class {';
			$css .= 'margin-right: auto;';
			$css .= '}';
			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu {';
			$css .= 'margin-right: auto;';
			$css .= '}';
		} else if(!empty($css_data['primary_menu_align_other']) && $css_data['primary_menu_align_other'] == 'center') {
			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu-class {';
			$css .= 'margin-right: auto;';
			$css .= 'margin-left: auto;';
			$css .= 'max-width: max-content;';
			$css .= '}';
			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu {';
			$css .= 'margin-right: auto;';
			$css .= 'margin-left: auto;';
			$css .= '}';
		} else if(!empty($css_data['primary_menu_align_other']) && $css_data['primary_menu_align_other'] == 'right') {
			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu-class {';
			$css .= 'margin-left: auto;';
			$css .= '}';
			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu {';
			$css .= 'margin-left: auto;';
			$css .= '}';
		}

		// Primary menu font on Home page
		if( !empty($css_data['primary_menu_font_front']) ) {
			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu a, .datebook-front-container .datebook-primary-menu-container .datebook-primary-menu .nav-link {';
			if( !empty($css_data['primary_menu_font_front']['size']) ) {
				$css .= 'font-size: ' . $css_data['primary_menu_font_front']['size'] . $css_data['primary_menu_font_front']['measure'] . ';';
			}
			if( !empty($css_data['primary_menu_font_front']['face']) ) {
				if ( array_key_exists( $css_data['primary_menu_font_front']['face'], $recognized_fonts ) ) {
					$css .= 'font-family: ' . $recognized_fonts[$css_data['primary_menu_font_front']['face']] . ';';
				}
			}
			if( !empty($css_data['primary_menu_font_front']['style']) ) {
				$css .= 'font-style: ' . $css_data['primary_menu_font_front']['style'] . ';';
			}
			if( !empty($css_data['primary_menu_font_front']['color']) ) {
				$css .= 'color: ' . $css_data['primary_menu_font_front']['color'] . ' !important;';
			}
			$css .= '}';
		}
		// Primary menu font on Other page
		if( !empty($css_data['primary_menu_font_other']) ) {
			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu a, .datebook-other-container .datebook-primary-menu-container .datebook-primary-menu .nav-link {';
			if( !empty($css_data['primary_menu_font_other']['size']) ) {
				$css .= 'font-size: ' . $css_data['primary_menu_font_other']['size'] . $css_data['primary_menu_font_other']['measure'] . ';';
			}
			if( !empty($css_data['primary_menu_font_other']['face']) ) {
				if ( array_key_exists( $css_data['primary_menu_font_other']['face'], $recognized_fonts ) ) {
					$css .= 'font-family: ' . $recognized_fonts[$css_data['primary_menu_font_other']['face']] . ';';
				}
			}
			if( !empty($css_data['primary_menu_font_other']['style']) ) {
				$css .= 'font-style: ' . $css_data['primary_menu_font_other']['style'] . ';';
			}
			if( !empty($css_data['primary_menu_font_other']['color']) ) {
				$css .= 'color: ' . $css_data['primary_menu_font_other']['color'] . ' !important;';
			}
			$css .= '}';
		}

		// Primary menu width on Home page
		if(!empty($css_data['primary_menu_bg_color_front'])){
			$css .= '.datebook-front-container .datebook-primary-menu-container {';
			$css .= 'background-color: ' . $css_data['primary_menu_bg_color_front'] . ';';
			$css .= '}';

			$css .= '.datebook-front-container #datebook-header .navbar-dark .dropdown-menu {';
			$css .= 'background-color: ' . $css_data['primary_menu_bg_color_front'] . ' !important;';
			$css .= '}';
		}
		// Primary menu width on Other page
		if(!empty($css_data['primary_menu_bg_color_other'])){
			$css .= '.datebook-other-container .datebook-primary-menu-container {';
			$css .= 'background-color: ' . $css_data['primary_menu_bg_color_other'] . ';';
			$css .= '}';

			$css .= '.datebook-other-container #datebook-header .navbar-dark .dropdown-menu {';
			$css .= 'background-color: ' . $css_data['primary_menu_bg_color_other'] . ' !important;';
			$css .= '}';
		}

		// Primary menu container border radius on Home page
		if( !empty($css_data['primary_menu_container_border_round_front']) || $css_data['primary_menu_container_border_round_front'] == 0 ) {
			$css .= '.datebook-front-container .datebook-primary-menu-container {';
			$css .= 'border-radius: ' . $css_data['primary_menu_container_border_round_front'] . 'rem;';
			$css .= '}';
		}
		// Primary menu container border radius on Other page
		if( !empty($css_data['primary_menu_container_border_round_other']) || $css_data['primary_menu_container_border_round_other'] == 0 ) {
			$css .= '.datebook-other-container .datebook-primary-menu-container {';
			$css .= 'border-radius: ' . $css_data['primary_menu_container_border_round_other'] . 'rem;';
			$css .= '}';
		}

		// Primary menu subcontainer border radius on Home page
		if( !empty($css_data['primary_menu_subcontainer_border_round_front']) || $css_data['primary_menu_subcontainer_border_round_front'] == 0 ) {
			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer {';
			$css .= 'border-radius: ' . $css_data['primary_menu_subcontainer_border_round_front'] . 'rem;';
			$css .= '}';
		}
		// Primary menu subcontainer border radius on Other page
		if( !empty($css_data['primary_menu_subcontainer_border_round_other']) || $css_data['primary_menu_subcontainer_border_round_other'] == 0 ) {
			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer {';
			$css .= 'border-radius: ' . $css_data['primary_menu_subcontainer_border_round_other'] . 'rem;';
			$css .= '}';
		}
		
		// Primary menu class background color on Home page
		if(!empty($css_data['primary_menu_class_bg_color_front'])) {
			$css .= '.datebook-front-container .datebook-primary-menu-subcontainer .datebook-primary-menu {';
			$css .= 'background-color: ' . $css_data['primary_menu_class_bg_color_front'] . ';';
			$css .= '}';

			$css .= '.datebook-front-container #datebook-header .navbar-dark .dropdown-menu {';
			$css .= 'background-color: ' . $css_data['primary_menu_class_bg_color_front'] . ' !important;';
			$css .= '}';
		}
		// Primary menu class background color on Other page
		if(!empty($css_data['primary_menu_class_bg_color_other'])) {
			$css .= '.datebook-other-container .datebook-primary-menu-subcontainer .datebook-primary-menu {';
			$css .= 'background-color: ' . $css_data['primary_menu_class_bg_color_other'] . ';';
			$css .= '}';

			$css .= '.datebook-other-container #datebook-header .navbar-dark .dropdown-menu {';
			$css .= 'background-color: ' . $css_data['primary_menu_class_bg_color_other'] . ' !important;';
			$css .= '}';
		}

		// Primary menu class border radius on Home page
		if( !empty($css_data['primary_menu_class_border_round_front']) || $css_data['primary_menu_class_border_round_front'] == 0 ) {
			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu {';
			$css .= 'border-radius: ' . $css_data['primary_menu_class_border_round_front'] . 'rem;';
			$css .= '}';
		}
		// Primary menu class border radius on Other page
		if( !empty($css_data['primary_menu_class_border_round_other']) || $css_data['primary_menu_class_border_round_other'] == 0 ) {
			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-primary-menu {';
			$css .= 'border-radius: ' . $css_data['primary_menu_class_border_round_other'] . 'rem;';
			$css .= '}';
		}

		// Primary menu toggler background color on Home page
		if( !empty($css_data['primary_menu_toggler_bg_color_front']) ) {
			$css .= '.datebook-front-container .datebook-primary-menu-container .navbar-toggler {';
			$css .= 'background-color: ' . $css_data['primary_menu_toggler_bg_color_front'] . ';';
			$css .= '}';
		}
		// Primary menu toggler background color on Home page
		if( !empty($css_data['primary_menu_toggler_bg_color_other']) ) {
			$css .= '.datebook-other-container .datebook-primary-menu-container .navbar-toggler {';
			$css .= 'background-color: ' . $css_data['primary_menu_toggler_bg_color_other'] . ';';
			$css .= '}';
		}

		// Primary menu margin on Home page
		if( !empty($css_data['primary_menu_margin_front']) ) {
			$css .= '.datebook-primary-menu-container .datebook-primary-menu {';
			if( !empty($css_data['primary_menu_margin_front']['margin_top']) ) {
				$css .= 'margin-top: ' . $css_data['primary_menu_margin_front']['margin_top'] . ';';
			}
			if( !empty($css_data['primary_menu_margin_front']['margin_right']) ) {
				$css .= 'margin-right: ' . $css_data['primary_menu_margin_front']['margin_right'] . ';';
			}
			if( !empty($css_data['primary_menu_margin_front']['margin_bottom']) ) {
				$css .= 'margin-bottom: ' . $css_data['primary_menu_margin_front']['margin_bottom'] . ';';
			}
			if( !empty($css_data['primary_menu_margin_front']['margin_left']) ) {
				$css .= 'margin-left: ' . $css_data['primary_menu_margin_front']['margin_left'] . ';';
			}
			$css .= '}';
		}

		// Primary menu padding on Home page
		if( !empty($css_data['primary_menu_padding_front']) ) {
			$css .= '.datebook-front-container .datebook-primary-menu {';
			if( !empty($css_data['primary_menu_padding_front']['padding_top']) ) {
				$css .= 'padding-top: ' . $css_data['primary_menu_padding_front']['padding_top'] . ';';
			}
			if( !empty($css_data['primary_menu_padding_front']['padding_right']) ) {
				$css .= 'padding-right: ' . $css_data['primary_menu_padding_front']['padding_right'] . ';';
			}
			if( !empty($css_data['primary_menu_padding_front']['padding_bottom']) ) {
				$css .= 'padding-bottom: ' . $css_data['primary_menu_padding_front']['padding_bottom'] . ';';
			}
			if( !empty($css_data['primary_menu_padding_front']['padding_left']) ) {
				$css .= 'padding-left: ' . $css_data['primary_menu_padding_front']['padding_left'] . ';';
			}
			$css .= '}';
		}
		// Primary menu padding on Other page
		if( !empty($css_data['primary_menu_padding_other']) ) {
			$css .= '.datebook-other-container .datebook-primary-menu {';
			if( !empty($css_data['primary_menu_padding_other']['padding_top']) ) {
				$css .= 'padding-top: ' . $css_data['primary_menu_padding_other']['padding_top'] . ';';
			}
			if( !empty($css_data['primary_menu_padding_other']['padding_right']) ) {
				$css .= 'padding-right: ' . $css_data['primary_menu_padding_other']['padding_right'] . ';';
			}
			if( !empty($css_data['primary_menu_padding_other']['padding_bottom']) ) {
				$css .= 'padding-bottom: ' . $css_data['primary_menu_padding_other']['padding_bottom'] . ';';
			}
			if( !empty($css_data['primary_menu_padding_other']['padding_left']) ) {
				$css .= 'padding-left: ' . $css_data['primary_menu_padding_other']['padding_left'] . ';';
			}
			$css .= '}';
		}

		// Logo align on Home page
		if(!empty($css_data['primary_menu_align_front'])) {
			$css .= '.datebook-front-container .datebook-logo-container {';
			$css .= 'text-align: ' . $css_data['primary_menu_align_front'] . ';';
			$css .= '}';
		}
		// Logo align on Other page
		if(!empty($css_data['primary_menu_align_other'])) {
			$css .= '.datebook-other-container .datebook-logo-container {';
			$css .= 'text-align: ' . $css_data['primary_menu_align_other'] . ';';
			$css .= '}';
		}

		// Primary menu subcontainer background color on Home page
		if(!empty($css_data['primary_menu_subcontainer_bg_color_front'])) {
			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer {';
			$css .= 'background-color: ' . $css_data['primary_menu_subcontainer_bg_color_front'] . ';';
			$css .= '}';

			$css .= '.datebook-front-container .datebook-primary-menu-container .dropdown-menu, .datebook-front-container #datebook-header .datebook-primary-menu-container .dropdown-menu .navbar-dark .dropdown-menu {';
			$css .= 'background-color: ' . $css_data['primary_menu_subcontainer_bg_color_front'] . ' !important;';
			$css .= 'border: none;';
			$css .= '}';
		}
		// Primary menu subcontainer background color on Other page
		if(!empty($css_data['primary_menu_subcontainer_bg_color_other'])) {
			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer {';
			$css .= 'background-color: ' . $css_data['primary_menu_subcontainer_bg_color_other'] . ';';
			$css .= '}';

			$css .= '.datebook-other-container .datebook-primary-menu-container .dropdown-menu, .datebook-other-container #datebook-header .datebook-primary-menu-container .dropdown-menu .navbar-dark .dropdown-menu {';
			$css .= 'background-color: ' . $css_data['primary_menu_subcontainer_bg_color_other'] . ' !important;';
			$css .= 'border: none;';
			$css .= '}';
		}


		/////////////////////////////////////////////////


		// User menu align on Home page
		if(!empty($css_data['user_menu_align_front']) && $css_data['user_menu_align_front'] == 'left') {
			$css .= '.datebook-front-container .datebook-user-navbar, .datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-userpanel-under {';
			$css .= 'margin-right: auto;';
			$css .= '}';
			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer datebook-register-links {';
			$css .= 'margin-right: auto;';
			$css .= '}';
		} else if(!empty($css_data['user_menu_align_front']) && $css_data['user_menu_align_front'] == 'center') {
			$css .= '.datebook-front-container .datebook-user-navbar, .datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-userpanel-under {';
			$css .= 'margin-right: auto;';
			$css .= 'margin-left: auto;';
			$css .= 'max-width: max-content;';
			$css .= '}';
			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer datebook-register-links {';
			$css .= 'margin-right: auto;';
			$css .= 'margin-left: auto;';
			$css .= '}';
		} else if(!empty($css_data['user_menu_align_front']) && $css_data['user_menu_align_front'] == 'right') {
			$css .= '.datebook-front-container .datebook-user-navbar, .datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-userpanel-under {';
			$css .= 'margin-left: auto;';
			$css .= '}';
			$css .= '.datebook-front-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer datebook-register-links {';
			$css .= 'margin-left: auto;';
			$css .= '}';
		}
		// User menu align on Other page
		if(!empty($css_data['user_menu_align_other']) && $css_data['user_menu_align_other'] == 'left') {
			$css .= '.datebook-other-container .datebook-user-navbar, .datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-userpanel-under {';
			$css .= 'margin-right: auto;';
			$css .= '}';
			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer datebook-register-links {';
			$css .= 'margin-right: auto;';
			$css .= '}';
		} else if(!empty($css_data['user_menu_align_other']) && $css_data['user_menu_align_other'] == 'center') {
			$css .= '.datebook-other-container .datebook-user-navbar, .datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-userpanel-under {';
			$css .= 'margin-right: auto;';
			$css .= 'margin-left: auto;';
			$css .= 'max-width: max-content;';
			$css .= '}';
			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer datebook-register-links {';
			$css .= 'margin-right: auto;';
			$css .= 'margin-left: auto;';
			$css .= '}';
		} else if(!empty($css_data['user_menu_align_other']) && $css_data['user_menu_align_other'] == 'right') {
			$css .= '.datebook-other-container .datebook-user-navbar, .datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer .datebook-userpanel-under {';
			$css .= 'margin-left: auto;';
			$css .= '}';
			$css .= '.datebook-other-container .datebook-primary-menu-container .datebook-primary-menu-subcontainer datebook-register-links {';
			$css .= 'margin-left: auto;';
			$css .= '}';
		}


		if( !empty($css_data['user_menu_font_front']) ) {
			$css .= '.datebook-front-container .datebook-user-navbar .nav-item .nav-link, .datebook-front-container .datebook-header-user-panel .navbar a, .datebook-front-container .datebook-userpanel-under .navbar .nav-item .nav-link, .datebook-front-container .datebook-userpanel-under .navbar .nav-item .dropdown-menu .dropdown-item, .datebook-front-container .navbar.datebook-user-navbar .dropdown-menu a {';
			if( !empty($css_data['user_menu_font_front']['size']) ) {
				$css .= 'font-size: ' . $css_data['user_menu_font_front']['size'] . $css_data['user_menu_font_front']['measure'] . ';';
			}
			if( !empty($css_data['user_menu_font_front']['face']) ) {
				if ( array_key_exists( $css_data['user_menu_font_front']['face'], $recognized_fonts ) ) {
					$css .= 'font-family: ' . $recognized_fonts[$css_data['user_menu_font_front']['face']] . ';';
				}
			}
			if( !empty($css_data['user_menu_font_front']['style']) ) {
				$css .= 'font-style: ' . $css_data['user_menu_font_front']['style'] . ';';
			}
			if( !empty($css_data['user_menu_font_front']['color']) ) {
				$css .= 'color: ' . $css_data['user_menu_font_front']['color'] . ';';
			}
			$css .= '}';

			$css .= '.datebook-front-container #datebook-header .navbar .datebook-register-links .dropdown-menu a, .datebook-front-container #datebook-header .navbar .datebook-register-links .dropdown-menu a:focus, .datebook-front-container #datebook-header .navbar .datebook-register-links .dropdown-menu a:hover {';
			if( !empty($css_data['user_menu_font_front']['face']) ) {
				if ( array_key_exists( $css_data['user_menu_font_front']['face'], $recognized_fonts ) ) {
					$css .= 'font-family: ' . $recognized_fonts[$css_data['user_menu_font_front']['face']] . ';';
				}
			}
			if( !empty($css_data['user_menu_font_front']['style']) ) {
				$css .= 'font-style: ' . $css_data['user_menu_font_front']['style'] . ';';
			}
			if( !empty($css_data['user_menu_font_front']['color']) ) {
				$css .= 'color: ' . $css_data['user_menu_font_front']['color'] . ';';
			}
			$css .= '}';
		}

		if( !empty($css_data['user_menu_font_other']) ) {
			$css .= '.datebook-other-container .datebook-user-navbar .nav-item .nav-link, .datebook-other-container .datebook-header-user-panel .navbar a, .datebook-other-container .datebook-userpanel-under .navbar .nav-item .nav-link, .datebook-other-container .datebook-userpanel-under .navbar .nav-item .dropdown-menu .dropdown-item, .datebook-other-container .navbar.datebook-user-navbar .dropdown-menu a {';
			if( !empty($css_data['user_menu_font_other']['size']) ) {
				$css .= 'font-size: ' . $css_data['user_menu_font_other']['size'] . $css_data['user_menu_font_other']['measure'] . ';';
			}
			if( !empty($css_data['user_menu_font_other']['face']) ) {
				if ( array_key_exists( $css_data['user_menu_font_other']['face'], $recognized_fonts ) ) {
					$css .= 'font-family: ' . $recognized_fonts[$css_data['user_menu_font_other']['face']] . ';';
				}
			}
			if( !empty($css_data['user_menu_font_other']['style']) ) {
				$css .= 'font-style: ' . $css_data['user_menu_font_other']['style'] . ';';
			}
			if( !empty($css_data['user_menu_font_other']['color']) ) {
				$css .= 'color: ' . $css_data['user_menu_font_other']['color'] . ';';
			}
			$css .= '}';

			$css .= '.datebook-other-container #datebook-header .navbar .datebook-register-links .dropdown-menu a, .datebook-other-container #datebook-header .navbar .datebook-register-links .dropdown-menu a:focus, .datebook-other-container #datebook-header .navbar .datebook-register-links .dropdown-menu a:hover {';
			if( !empty($css_data['user_menu_font_other']['face']) ) {
				if ( array_key_exists( $css_data['user_menu_font_other']['face'], $recognized_fonts ) ) {
					$css .= 'font-family: ' . $recognized_fonts[$css_data['user_menu_font_other']['face']] . ';';
				}
			}
			if( !empty($css_data['user_menu_font_other']['style']) ) {
				$css .= 'font-style: ' . $css_data['user_menu_font_other']['style'] . ';';
			}
			if( !empty($css_data['user_menu_font_other']['color']) ) {
				$css .= 'color: ' . $css_data['user_menu_font_other']['color'] . ';';
			}
			$css .= '}';
		}

		// User menu background color on Home page
		if( !empty($css_data['user_menu_bg_color_front']) ) {
			$css .= '.datebook-front-container .datebook-user-navbar, .datebook-front-container .datebook-userpanel-under, .datebook-front-container .datebook-userpanel-inheader {';
			$css .= 'background-color: ' . $css_data['user_menu_bg_color_front'] . ' !important;';
			$css .= 'border: none;';
			$css .= '}';
		}
		// User menu background color on Other page
		if( !empty($css_data['user_menu_bg_color_other']) ) {
			$css .= '.datebook-other-container .datebook-user-navbar, .datebook-other-container .datebook-userpanel-under, .datebook-other-container .datebook-userpanel-inheader {';
			$css .= 'background-color: ' . $css_data['user_menu_bg_color_other'] . ' !important;';
			$css .= 'border: none;';
			$css .= '}';
		}

		// User menu toggler background color on Home page
		if( !empty($css_data['user_menu_toggler_bg_color_front']) ) {
			$css .= '.datebook-front-container #datebook-header .datebook-userpanel-inheader .navbar-dark .datebook-front-container .datebook-register-links .dropdown-menu, .datebook-front-container .datebook-userpanel-inheader .datebook-register-links .dropdown-menu, .datebook-front-container #datebook-header .navbar-dark .datebook-register-links .dropdown-menu, .datebook-front-container .datebook-user-navbar .datebook-register-links .dropdown-menu {';
			$css .= 'background-color: ' . $css_data['user_menu_toggler_bg_color_front'] . ' !important;';
			$css .= 'border: none;';
			$css .= '}';
		}
		// User menu toggler background color on Other page
		if( !empty($css_data['user_menu_toggler_bg_color_other']) ) {
			$css .= '.datebook-other-container #datebook-header .datebook-userpanel-inheader .navbar-dark .datebook-register-links .dropdown-menu, .datebook-other-container .datebook-userpanel-inheader .datebook-register-links .dropdown-menu, .datebook-other-container #datebook-header .navbar-dark .datebook-register-links .dropdown-menu, .datebook-other-container .datebook-user-navbar .datebook-register-links .dropdown-menu {';
			$css .= 'background-color: ' . $css_data['user_menu_toggler_bg_color_other'] . ' !important;';
			$css .= 'border: none;';
			$css .= '}';
			
			$css .= '.datebook-user-navbar ul ul:before {';
			$css .= 'border-color: transparent transparent ' . $css_data['user_menu_toggler_bg_color_other'] . ' transparent;';
			$css .= '}';

			
		}


		if( !empty($css_data['user_menu_margin_front']) ) {
			$css .= '.datebook-header-user-panel .navbar-nav {';
			if( !empty($css_data['user_menu_margin_front']['margin_top']) ) {
				$css .= 'margin-top: ' . $css_data['user_menu_margin_front']['margin_top'] . ';';
			}
			if( !empty($css_data['user_menu_margin_front']['margin_right']) ) {
				$css .= 'margin-right: ' . $css_data['user_menu_margin_front']['margin_right'] . ';';
			}
			if( !empty($css_data['user_menu_margin_front']['margin_bottom']) ) {
				$css .= 'margin-bottom: ' . $css_data['user_menu_margin_front']['margin_bottom'] . ';';
			}
			if( !empty($css_data['user_menu_margin_front']['margin_left']) ) {
				$css .= 'margin-left: ' . $css_data['user_menu_margin_front']['margin_left'] . ';';
			}
			$css .= '}';
		}
		if( !empty($css_data['user_menu_padding_front']) ) {
			$css .= '.datebook-header-user-panel .navbar-nav {';
			if( !empty($css_data['user_menu_padding_front']['padding_top']) ) {
				$css .= 'padding-top: ' . $css_data['user_menu_padding_front']['padding_top'] . ';';
			}
			if( !empty($css_data['user_menu_padding_front']['padding_right']) ) {
				$css .= 'padding-right: ' . $css_data['user_menu_padding_front']['padding_right'] . ';';
			}
			if( !empty($css_data['user_menu_padding_front']['padding_bottom']) ) {
				$css .= 'padding-bottom: ' . $css_data['user_menu_padding_front']['padding_bottom'] . ';';
			}
			if( !empty($css_data['user_menu_padding_front']['padding_left']) ) {
				$css .= 'padding-left: ' . $css_data['user_menu_padding_front']['padding_left'] . ';';
			}
			$css .= '}';
		}

		if( !empty($css_data['user_menu_margin_other']) ) {
			$css .= '.datebook-other-container .datebook-userpanel-under {';
			if( !empty($css_data['user_menu_margin_other']['margin_top']) ) {
				$css .= 'margin-top: ' . $css_data['user_menu_margin_other']['margin_top'] . ';';
			}
			if( !empty($css_data['user_menu_margin_other']['margin_right']) ) {
				$css .= 'margin-right: ' . $css_data['user_menu_margin_other']['margin_right'] . ';';
			}
			if( !empty($css_data['user_menu_margin_other']['margin_bottom']) ) {
				$css .= 'margin-bottom: ' . $css_data['user_menu_margin_other']['margin_bottom'] . ';';
			}
			if( !empty($css_data['user_menu_margin_other']['margin_left']) ) {
				$css .= 'margin-left: ' . $css_data['user_menu_margin_other']['margin_left'] . ';';
			}
			$css .= '}';
		}
		if( !empty($css_data['user_menu_padding_front']) ) {
			$css .= '.datebook-header-user-panel .navbar-nav {';
			if( !empty($css_data['user_menu_padding_front']['padding_top']) ) {
				$css .= 'padding-top: ' . $css_data['user_menu_padding_front']['padding_top'] . ';';
			}
			if( !empty($css_data['user_menu_padding_front']['padding_right']) ) {
				$css .= 'padding-right: ' . $css_data['user_menu_padding_front']['padding_right'] . ';';
			}
			if( !empty($css_data['user_menu_padding_front']['padding_bottom']) ) {
				$css .= 'padding-bottom: ' . $css_data['user_menu_padding_front']['padding_bottom'] . ';';
			}
			if( !empty($css_data['user_menu_padding_front']['padding_left']) ) {
				$css .= 'padding-left: ' . $css_data['user_menu_padding_front']['padding_left'] . ';';
			}
			$css .= '}';
		}


		// Custom Panels
		if ($css_data['datebook_custom_panels']) {

			$css .= '.datebook-custom-card.card {';

			$css .= 'width: 100%;';

			if( isset($css_data['panel_border_width']) ) {
				$css .= 'border-width:' . $css_data['panel_border_width'] . 'px;';
			}
			if( !empty($css_data['panel_border_style']) ) {
				$css .= 'border-style:' . $css_data['panel_border_style'] . ';';
			}
			if( !empty($css_data['panel_border_color']) ) {
				$css .= 'border-color:' . $css_data['panel_border_color'] . ';';
			}
			if( !empty($css_data['panel_border_round']) ) {
				if($css_data['panel_border_round'] == 0) {
					$css .= 'border-radius: 0;';
				} else {
					$css .= 'border-radius: ' . $css_data['panel_border_round'] . 'rem;';
				}
			}
			if( !empty($css_data['panel_container_margin']) ) {
				if( !empty($css_data['panel_container_margin']['margin_top']) ) {
					$css .= 'margin-top: ' . $css_data['panel_container_margin']['margin_top'] . ' !important;';
				}
				if( !empty($css_data['panel_container_margin']['margin_right']) ) {
					$css .= 'margin-right: ' . $css_data['panel_container_margin']['margin_right'] . ' !important;';
				}
				if( !empty($css_data['panel_container_margin']['margin_bottom']) ) {
					$css .= 'margin-bottom: ' . $css_data['panel_container_margin']['margin_bottom'] . ' !important;';
				}
				if( !empty($css_data['panel_container_margin']['margin_left']) ) {
					$css .= 'margin-left: ' . $css_data['panel_container_margin']['margin_left'] . ' !important;';
				}
			}
			if( !empty($css_data['panel_text_color']) ) {
				$css .= 'color:' . $css_data['panel_text_color'] . ';';
			}
			if( !empty($css_data['panel_background_color']) ) {
				$css .= 'background-color:' . $css_data['panel_background_color'] . ';';
			}
			if( !empty($css_data['panel_border_shadow']) ) {
				
				$css .= $this->shadows_list($css_data['panel_border_shadow']);
				
			}

			$css .= '}';

			if( !empty($css_data['panel_border_round']) ) {
				$css .= '.datebook-custom-card.card .card-header:first-child {';
				if($css_data['panel_border_round'] == 0) {
					$css .= 'border-radius: 0;';
				} else {
					$css .= 'border-radius: calc(' . $css_data['panel_border_round'] . 'rem - 1px) calc(' . $css_data['panel_border_round'] . 'rem - 1px) 0 0;';
				}
				$css .= '}';

				
				if($css_data['panel_border_round'] == 0) {
					$css .= '';
				} else {
					$css .= '.alert {';
					$css .= 'border-radius: ' . $css_data['panel_border_round'] . 'rem;';
					$css .= '}';
				}
				
			}

			$css .= '.datebook-custom-card.card .card-header {';
			if( !empty($css_data['panel_header_text_color']) ) {
				$css .= 'color:' . $css_data['panel_header_text_color'] . ';';
			}
			if( !empty($css_data['panel_header_bg_color']) ) {
				$css .= 'background-color:' . $css_data['panel_header_bg_color'] . ';';
			}
			$css .= '}';

			if( !empty($css_data['panel_container_padding']) ) {
				$css .= '.datebook-custom-card.card .card-body {';
				if( !empty($css_data['panel_container_padding']['padding_top']) ) {
					$css .= 'padding-top: ' . $css_data['panel_container_padding']['padding_top'] . ' !important;';
				}
				if( !empty($css_data['panel_container_padding']['padding_right']) ) {
					$css .= 'padding-right: ' . $css_data['panel_container_padding']['padding_right'] . ' !important;';
				}
				if( !empty($css_data['panel_container_padding']['padding_bottom']) ) {
					$css .= 'padding-bottom: ' . $css_data['panel_container_padding']['padding_bottom'] . ' !important;';
				}
				if( !empty($css_data['panel_container_padding']['padding_left']) ) {
					$css .= 'padding-left: ' . $css_data['panel_container_padding']['padding_left'] . ' !important;';
				}
				$css .= '}';
			}

		}


		// Search Form on the Home page
		$css .= '.datebook-search-panel .datebook-search-front {';

		if( !empty($css_data['gradientcolor_searchform_front']) ) {
			$css .= 'position: inherit;';
			$css .= 'border: none;';
			$css .= 'background: rgba(0, 0, 0, 0.5);';
			$css .= 'background: -webkit-gradient(linear, 38deg, color-stop(0%, rgba(0, 0, 0, 0.6)), color-stop(50%, #000000), color-stop(100%, #595959));';
			$css .= 'background: -ms-' . $css_data['gradientcolor_searchform_front'] . ';';
			$css .= 'background: -moz-' . $css_data['gradientcolor_searchform_front'] . ';';
			$css .= 'background: -o-' . $css_data['gradientcolor_searchform_front'] . ';';
			$css .= 'background: -webkit-' . $css_data['gradientcolor_searchform_front'] . ';';
			$css .= 'background: ' . $css_data['gradientcolor_searchform_front'] . ';';
		}

		if( !empty($css_data['gradientcolor_searchform_front']) ) {
			if(isset($css_data['searchform_border_width_front']) ) {
				$css .= 'border-width:' . $css_data['searchform_border_width_front'] . 'px;';
			}
			if(isset($css_data['searchform_border_style_front']) ) {
				$css .= 'border-style:' . $css_data['searchform_border_style_front'] . ';';
			}
			if(isset($css_data['searchform_border_color_front']) ) {
				$css .= 'border-color:' . $css_data['searchform_border_color_front'] . ';';
			}
			if(isset($css_data['searchform_border_round_front']) ) {
				if($css_data['searchform_border_round_front'] == 0) {
					$css .= 'border-radius: 0;';
				} else {
					$css .= 'border-radius: ' . $css_data['searchform_border_round_front'] . 'rem;';
				}
			}
		}

		if( !empty($css_data['searchform_border_shadow_front']) ) {
			
			$css .= $this->shadows_list($css_data['searchform_border_shadow_front']);
			
		}

		// Margin
		if( !empty($css_data['searchform_margin_front']) ) {
			if( !empty($css_data['searchform_margin_front']['margin_top']) ) {
				$css .= 'margin-top: ' . $css_data['searchform_margin_front']['margin_top'] . ' !important;';
			}
			if( !empty($css_data['searchform_margin_front']['margin_right']) ) {
				$css .= 'margin-right: ' . $css_data['searchform_margin_front']['margin_right'] . ' !important;';
			}
			if( !empty($css_data['searchform_margin_front']['margin_bottom']) ) {
				$css .= 'margin-bottom: ' . $css_data['searchform_margin_front']['margin_bottom'] . ' !important;';
			}
			if( !empty($css_data['searchform_margin_front']['margin_left']) ) {
				$css .= 'margin-left: ' . $css_data['searchform_margin_front']['margin_left'] . ' !important;';
			}
		}

		// Padding
		if( !empty($css_data['searchform_padding_front']) ) {
			if( !empty($css_data['searchform_padding_front']['padding_top']) ) {
				$css .= 'padding-top: ' . $css_data['searchform_padding_front']['padding_top'] . ' !important;';
			}
			if( !empty($css_data['searchform_padding_front']['padding_right']) ) {
				$css .= 'padding-right: ' . $css_data['searchform_padding_front']['padding_right'] . ' !important;';
			}
			if( !empty($css_data['searchform_padding_front']['padding_bottom']) ) {
				$css .= 'padding-bottom: ' . $css_data['searchform_padding_front']['padding_bottom'] . ' !important;';
			}
			if( !empty($css_data['searchform_padding_front']['padding_left']) ) {
				$css .= 'padding-left: ' . $css_data['searchform_padding_front']['padding_left'] . ' !important;';
			}
		}
		
		$css .= '}';

		// Card header
		$css .= '.datebook-search-panel .datebook-search-front .card-header {';
		if( !empty($css_data['searchform_header_text_color_front']) ) {
			$css .= 'color:' . $css_data['searchform_header_text_color_front'] . ';';
		}
		if( !empty($css_data['searchform_header_bg_color_front']) ) {
			$css .= 'background-color:' . $css_data['searchform_header_bg_color_front'] . ';';
		}
		$css .= '}';

		// Text color
		if( !empty($css_data['searchform_text_color_front']) ) {
			$css .= '.datebook-search-panel .datebook-search-front label.datebook-label-title {';
			$css .= 'color:' . $css_data['searchform_text_color_front'] . ';';
			$css .= '}';
		}

		// Gender button front
		if(!empty($css_data['searchform_gender_bgcolor_front'])){
			$css .= '.datebook-search-panel .datebook-search-front .datebook-search-gender:hover, .datebook-search-panel .datebook-search-front .datebook-search-gender:not([disabled]):not(.disabled).active, .datebook-search-panel .datebook-search-front .datebook-search-gender:not(:disabled):not(.disabled).active:focus, .datebook-search-panel .datebook-search-front .datebook-search-gender:not(:disabled):not(.disabled):active:focus, .datebook-search-panel .datebook-search-front .datebook-search-gender.dropdown-toggle:focus {';
			$css .= 'background-color: ' . $this->adjustBrightness($css_data['searchform_gender_bgcolor_front'], -0.4) . ' !important;';
			$css .= '}';

			$css .= '.datebook-search-panel .datebook-search-front .btn.focus, .datebook-search-panel .datebook-search-front .btn:focus {';
			$css .= 'box-shadow: none !important;';
			$css .= '}';

			$css .= '.datebook-search-panel .datebook-search-front .datebook-search-gender {';
			$css .= 'background-color: ' . $css_data['searchform_gender_bgcolor_front'] . ' !important;';
			$css .= 'color: ' . $css_data['searchform_gender_color_front'] . ';';
			$css .= '}';
		}

		// Submit button front
		if(!empty($css_data['searchform_submit_bgcolor_front'])){
			$css .= '.datebook-search-panel .datebook-search-front .datebook-search-submit:hover, .datebook-search-panel .datebook-search-front .datebook-search-submit.active {';
			$css .= 'background-color: ' . $this->adjustBrightness($css_data['searchform_submit_bgcolor_front'], -0.2) . ' !important;';
			$css .= '}';

			$css .= '.datebook-search-panel .datebook-search-front .datebook-search-submit {';
			$css .= 'background-color: ' . $css_data['searchform_submit_bgcolor_front'] . ' !important;';
			$css .= 'color: ' . $css_data['searchform_submit_color_front'] . ';';
			$css .= '}';
		}


		// Search form on the Search page
		$css .= '.datebook-search-panel .datebook-search-other {';

		if(isset($css_data['searchform_border_width_other']) ) {
			$css .= 'border-width:' . $css_data['searchform_border_width_other'] . 'px;';
		}
		if(isset($css_data['searchform_border_style_other']) ) {
			$css .= 'border-style:' . $css_data['searchform_border_style_other'] . ';';
		}
		if(isset($css_data['searchform_border_color_other']) ) {
			$css .= 'border-color:' . $css_data['searchform_border_color_other'] . ';';
		}
		if(isset($css_data['searchform_border_round_other']) ) {
			if($css_data['searchform_border_round_other'] == 0) {
				$css .= 'border-radius: 0;';
			} else {
				$css .= 'border-radius: ' . $css_data['searchform_border_round_other'] . 'rem;';
			}
		}

		if( !empty($css_data['searchform_background_color_other']) ) {
			$css .= 'background-color:' . $css_data['searchform_background_color_other'] . ';';
		}

		if( !empty($css_data['searchform_border_shadow_other']) ) {
			
			$css .= $this->shadows_list($css_data['searchform_border_shadow_other']);
			
		}

		// Margin
		if( !empty($css_data['searchform_margin_other']) ) {
			if( !empty($css_data['searchform_margin_other']['margin_top']) ) {
				$css .= 'margin-top: ' . $css_data['searchform_margin_other']['margin_top'] . ' !important;';
			}
			if( !empty($css_data['searchform_margin_other']['margin_right']) ) {
				$css .= 'margin-right: ' . $css_data['searchform_margin_other']['margin_right'] . ' !important;';
			}
			if( !empty($css_data['searchform_margin_other']['margin_bottom']) ) {
				$css .= 'margin-bottom: ' . $css_data['searchform_margin_other']['margin_bottom'] . ' !important;';
			}
			if( !empty($css_data['searchform_margin_other']['margin_left']) ) {
				$css .= 'margin-left: ' . $css_data['searchform_margin_other']['margin_left'] . ' !important;';
			}
		}

		// Padding
		if( !empty($css_data['searchform_padding_other']) ) {
			if( !empty($css_data['searchform_padding_other']['padding_top']) ) {
				$css .= 'padding-top: ' . $css_data['searchform_padding_other']['padding_top'] . ' !important;';
			}
			if( !empty($css_data['searchform_padding_other']['padding_right']) ) {
				$css .= 'padding-right: ' . $css_data['searchform_padding_other']['padding_right'] . ' !important;';
			}
			if( !empty($css_data['searchform_padding_other']['padding_bottom']) ) {
				$css .= 'padding-bottom: ' . $css_data['searchform_padding_other']['padding_bottom'] . ' !important;';
			}
			if( !empty($css_data['searchform_padding_other']['padding_left']) ) {
				$css .= 'padding-left: ' . $css_data['searchform_padding_other']['padding_left'] . ' !important;';
			}
		}

		$css .= '}';

		// Card header
		$css .= '.datebook-search-panel .datebook-search-other .card-header {';
		if( !empty($css_data['searchform_header_text_color_other']) ) {
			$css .= 'color:' . $css_data['searchform_header_text_color_other'] . ';';
		}
		if( !empty($css_data['searchform_header_bg_color_other']) ) {
			$css .= 'background-color:' . $css_data['searchform_header_bg_color_other'] . ';';
		}
		$css .= '}';

		// Text color
		if( !empty($css_data['searchform_text_color_other']) ) {
			$css .= '.datebook-search-panel .datebook-search-other label.datebook-label-title {';
			$css .= 'color:' . $css_data['searchform_text_color_other'] . ';';
			$css .= '}';
		}

		// Gender button other
		if(!empty($css_data['searchform_gender_bgcolor_other'])){
			$css .= '.datebook-search-panel .datebook-search-other .datebook-search-gender:hover, .datebook-search-panel .datebook-search-other .datebook-search-gender:not([disabled]):not(.disabled).active, .datebook-search-panel .datebook-search-other .datebook-search-gender:not(:disabled):not(.disabled).active:focus, .datebook-search-panel .datebook-search-other .datebook-search-gender:not(:disabled):not(.disabled):active:focus, .show > .datebook-search-panel .datebook-search-other .datebook-search-gender.dropdown-toggle:focus {';
			$css .= 'background-color: ' . $this->adjustBrightness($css_data['searchform_gender_bgcolor_other'], -0.4) . ' !important;';
			$css .= '}';

			$css .= '.datebook-search-panel .datebook-search-other .btn.focus, .datebook-search-panel .datebook-search-other .btn:focus {';
			$css .= 'box-shadow: none !important;';
			$css .= '}';

			$css .= '.datebook-search-panel .datebook-search-other .datebook-search-gender {';
			$css .= 'background-color: ' . $css_data['searchform_gender_bgcolor_other'] . ' !important;';
			$css .= 'color: ' . $css_data['searchform_gender_color_other'] . ';';
			$css .= '}';
		}

		// Submit button other
		if(!empty($css_data['searchform_submit_bgcolor_other'])){
			$css .= '.datebook-search-panel .datebook-search-other .datebook-search-submit:hover, .datebook-search-panel .datebook-search-other .datebook-search-submit.active {';
			$css .= 'background-color: ' . $this->adjustBrightness($css_data['searchform_submit_bgcolor_other'], -0.2) . ' !important;';
			$css .= '}';

			$css .= '.datebook-search-panel .datebook-search-other .datebook-search-submit {';
			$css .= 'background-color: ' . $css_data['searchform_submit_bgcolor_other'] . ' !important;';
			$css .= 'color: ' . $css_data['searchform_submit_color_other'] . ';';
			$css .= '}';
		}


		// Padding
		if( !empty($css_data['searchform_padding_other']) ) {
			$css .= '.datebook-search-panel .datebook-search-other .card-body {';
			if( !empty($css_data['searchform_padding_other']['padding_top']) ) {
				$css .= 'padding-top: ' . $css_data['searchform_padding_other']['padding_top'] . ' !important;';
			}
			if( !empty($css_data['searchform_padding_other']['padding_right']) ) {
				$css .= 'padding-right: ' . $css_data['searchform_padding_other']['padding_right'] . ' !important;';
			}
			if( !empty($css_data['searchform_padding_other']['padding_bottom']) ) {
				$css .= 'padding-bottom: ' . $css_data['searchform_padding_other']['padding_bottom'] . ' !important;';
			}
			if( !empty($css_data['searchform_padding_other']['padding_left']) ) {
				$css .= 'padding-left: ' . $css_data['searchform_padding_other']['padding_left'] . ' !important;';
			}
			$css .= '}';
		}


		// Custom Profiles on Search page
		if ($css_data['datebook_profiles_on_search']) {

			$css .= '.datebook-search-profiles .card {';

			if( !empty($css_data['profiles_on_search_text_align']) ) {
				switch($css_data['profiles_on_search_text_align']) {
					case 'left' : $css .= 'text-align: left;'; break;
					case 'center' : $css .= 'text-align: center;'; break;
					case 'right' : $css .= 'text-align: right;'; break;
					case 'justify' : $css .= 'text-align: justify;'; break;
					default: $css .= ''; break;
				}
			}
			if( !empty($css_data['profiles_on_search_border_width']) ) {
				$css .= 'border-width:' . $css_data['profiles_on_search_border_width'] . 'px;';
			}
			if( !empty($css_data['profiles_on_search_border_style']) ) {
				$css .= 'border-style:' . $css_data['profiles_on_search_border_style'] . ';';
			}
			if( !empty($css_data['profiles_on_search_border_color']) ) {
				$css .= 'border-color:' . $css_data['profiles_on_search_border_color'] . ';';
			}
			if( !empty($css_data['profiles_on_search_border_round']) || $css_data['profiles_on_search_border_round'] == 0 ) {
				if($css_data['profiles_on_search_border_round'] == 0) {
					$css .= 'border-radius: 0;';
				} else {

					if($css_data['profiles_on_search_border_round'] <= 1){
						$css_data_border_round = '0.1';
					} else if($css_data['profiles_on_search_border_round'] <= 2){
						$css_data_border_round = '0.5';
					} else if($css_data['profiles_on_search_border_round'] <= 3){
						$css_data_border_round = $css_data['profiles_on_search_border_round'] / 3;
					} else if($css_data['profiles_on_search_border_round'] >= 6){
						$css_data_border_round = '3';
					} else {
						$css_data_border_round = $css_data['profiles_on_search_border_round'] / 2;
					}
					
					$css .= 'border-radius: ' . $css_data['profiles_on_search_border_round'] . 'rem;';
					$css .= 'padding-bottom: ' . $css_data_border_round . 'rem;';
				}
			}
			if( !empty($css_data['profiles_on_search_background_color']) ) {
				$css .= 'background-color:' . $css_data['profiles_on_search_background_color'] . ';';
			}
			if( !empty($css_data['profiles_on_search_border_shadow']) ) {
				
				$css .= $this->shadows_list($css_data['profiles_on_search_border_shadow']);
				
			}

			if( !empty($css_data['profiles_on_search_margin']) ) {
				if( !empty($css_data['profiles_on_search_margin']['margin_top']) ) {
					$css .= 'margin-top: ' . $css_data['profiles_on_search_margin']['margin_top'] . ' !important;';
				}
				if( !empty($css_data['profiles_on_search_margin']['margin_right']) ) {
					$css .= 'margin-right: ' . $css_data['profiles_on_search_margin']['margin_right'] . ' !important;';
				}
				if( !empty($css_data['profiles_on_search_margin']['margin_bottom']) ) {
					$css .= 'margin-bottom: ' . $css_data['profiles_on_search_margin']['margin_bottom'] . ' !important;';
				}
				if( !empty($css_data['profiles_on_search_margin']['margin_left']) ) {
					$css .= 'margin-left: ' . $css_data['profiles_on_search_margin']['margin_left'] . ' !important;';
				}
			}

			$css .= '}';
			
			$css .= '.datebook-profiles-list.datebook-hide .card {';
			$css .= 'padding-bottom: 0';
			$css .= '}';

			if( !empty($css_data['profiles_on_search_border_round']) || $css_data['profiles_on_search_border_round'] == 0 ) {
				$css .= '.datebook-profile-inline-caption, .card-img-top {';
				if($css_data['profiles_on_search_border_round'] == 0) {
					$css .= 'border-radius: 0;';
				} else {
					$css .= 'border-top-left-radius: calc(' . $css_data['profiles_on_search_border_round'] . 'rem - 1px);';
					$css .= 'border-top-right-radius: calc(' . $css_data['profiles_on_search_border_round'] . 'rem - 1px);';
				}
				$css .= '}';

				$css .= '.datebook-hide .datebook-profile-inline-caption, .datebook-hide .card-img-top {';
				if($css_data['profiles_on_search_border_round'] == 0) {
					$css .= 'border-radius: 0;';
				} else {
					$css .= 'border-bottom-left-radius: calc(' . $css_data['profiles_on_search_border_round'] . 'rem - 1px);';
					$css .= 'border-bottom-right-radius: calc(' . $css_data['profiles_on_search_border_round'] . 'rem - 1px);';
				}
				$css .= '}';
				
				
				if($css_data['profiles_on_search_border_round'] == 0) {
					$css .= '.datebook-profile-caption .datebook-top-label, .datebook-profile-caption .datebook-new-label {';
					$css .= 'border-radius: 0;';
					$css .= 'z-index: 10';
					$css .= '}';
				} else {
					$css .= '.datebook-profile-caption .datebook-top-label {';
					$css .= 'border-radius: calc(' . $css_data['profiles_on_search_border_round'] . 'rem - 10%);';
					$css .= 'top: 4px;';
					$css .= 'left: 4px;';
					$css .= 'z-index: 10';
					$css .= '}';

					$css .= '.datebook-profile-caption .datebook-new-label {';
					$css .= 'border-radius: calc(' . $css_data['profiles_on_search_border_round'] . 'rem - 10%);';
					$css .= 'top: 4px;';
					$css .= 'right: 4px;';
					$css .= '}';

					$css .= '.datebook-hide .datebook-profile-caption .datebook-profile-comment-caption {';
					$css .= 'border-radius: calc(' . $css_data['profiles_on_search_border_round'] . 'rem - 10%);';
					$css .= 'bottom: 4px;';
					$css .= 'left: 4px;';
					$css .= '}';

					$css .= '.datebook-hide .datebook-profile-caption .datebook-profile-pictures-caption {';
					$css .= 'border-radius: calc(' . $css_data['profiles_on_search_border_round'] . 'rem - 10%);';
					$css .= 'bottom: 4px;';
					$css .= 'right: 4px;';
					$css .= '}';

				}
				
			}

			
			if( !empty($css_data['profiles_on_search_header_text_color']) ) {
				$css .= '.datebook-search-profiles .card .card-title a {';
				$css .= 'color:' . $css_data['profiles_on_search_header_text_color'] . ';';
				$css .= '}';
			}

			if( !empty($css_data['profiles_on_search_padding']) ) {
				$css .= '.datebook-search-panel .datebook-profiles-list .card .card-body {';
				if( !empty($css_data['profiles_on_search_padding']['padding_top']) ) {
					$css .= 'padding-top: ' . $css_data['profiles_on_search_padding']['padding_top'] . ' !important;';
				}
				if( !empty($css_data['profiles_on_search_padding']['padding_right']) ) {
					$css .= 'padding-right: ' . $css_data['profiles_on_search_padding']['padding_right'] . ' !important;';
				}
				if( !empty($css_data['profiles_on_search_padding']['padding_bottom']) ) {
					$css .= 'padding-bottom: ' . $css_data['profiles_on_search_padding']['padding_bottom'] . ' !important;';
				}
				if( !empty($css_data['profiles_on_search_padding']['padding_left']) ) {
					$css .= 'padding-left: ' . $css_data['profiles_on_search_padding']['padding_left'] . ' !important;';
				}
				$css .= '}';
			}

			if( !empty($css_data['profiles_on_search_text_color']) ) {
				$css .= '.datebook-search-profiles .card .card-body .card-text {';
				$css .= 'color:' . $css_data['profiles_on_search_text_color'] . ';';
				$css .= '}';
			}
			
		}


			// Featured profiles
			if( !empty($css_data['featured_header_text_color_front']) ) {
				$css .= '.datebook-front-container .card.datebook-featured-profile .card-body .card-title a {';
				$css .= 'color:' . $css_data['featured_header_text_color_front'] . ';';
				$css .= '}';
			}

			if( !empty($css_data['featured_body_text_color_front']) ) {
				$css .= '.datebook-front-container .card.datebook-featured-profile .card-body .card-text {';
				$css .= 'color:' . $css_data['featured_body_text_color_front'] . ';';
				$css .= '}';
			}

			if( !empty($css_data['featured_background_color_front']) ) {
				$css .= '.datebook-front-container .card.datebook-featured-profile .card-body {';
				$css .= 'background-color:' . $css_data['featured_background_color_front'] . ';';
				$css .= '}';
			}

			if( !empty($css_data['featured_round_front']) ) {
				$css .= '#datebook-featured-front .datebook-featured-profile {';
				$css .= 'border-radius: ' . $css_data['featured_round_front'] . 'rem !important;';
				$css .= '}';

				$css .= '#datebook-featured-front .datebook-featured-profile.datebook-featured-compact .datebook-profile-comment-caption {';
				$css .= 'border-radius: calc(' . $css_data['featured_round_other'] . 'rem - 10%);';
				$css .= '}';

				$css .= '#datebook-featured-front .datebook-featured-profile.datebook-featured-compact .datebook-profile-pictures-caption {';
				$css .= 'border-radius: calc(' . $css_data['featured_round_other'] . 'rem - 1%);';
				$css .= '}';

				$css .= '#datebook-featured-front .datebook-featured-profile.datebook-featured-full img {';
				$css .= 'border-radius: calc(' . $css_data['featured_round_front'] . 'rem - 1%); border-bottom-left-radius: 0; border-bottom-right-radius: 0;';
				$css .= '}';

				$css .= '#datebook-featured-front .datebook-featured-profile.datebook-featured-compact img {';
				$css .= 'border-radius: calc(' . $css_data['featured_round_front'] . 'rem - 1%);';
				$css .= '}';

				$css .= '#datebook-featured-front .card.datebook-featured-profile.datebook-featured-full .card-body {';
				$css .= 'border-radius: calc(' . $css_data['featured_round_front'] . 'rem - 1%); border-top-left-radius: 0; border-top-right-radius: 0;';
				$css .= '}';
			}


			// Featured profile padding body
			if( !empty($css_data['featured_shadow_front']) ) {

				$css .= '.datebook-front-container .datebook-featured-profile {';
				
				$css .= $this->shadows_list($css_data['featured_shadow_front']);

				$css .= '}';

			}
			

			if( !empty($css_data['featured_header_text_color_other']) ) {
				$css .= '.datebook-other-container .card.datebook-featured-profile .card-body .card-title a {';
				$css .= 'color:' . $css_data['featured_header_text_color_other'] . ';';
				$css .= '}';
			}

			if( !empty($css_data['featured_body_text_color_other']) ) {
				$css .= '.datebook-other-container .card.datebook-featured-profile .card-body .card-text {';
				$css .= 'color:' . $css_data['featured_body_text_color_other'] . ';';
				$css .= '}';
			}

			if( !empty($css_data['featured_background_color_other']) ) {
				$css .= '.datebook-other-container .card.datebook-featured-profile .card-body {';
				$css .= 'background-color:' . $css_data['featured_background_color_other'] . ';';
				$css .= '}';
			}

			if( !empty($css_data['featured_round_other']) ) {
				$css .= '.datebook-other-container .datebook-featured-profile {';
				$css .= 'border-radius: ' . $css_data['featured_round_other'] . 'rem !important;';
				$css .= '}';

				$css .= '.datebook-other-container .datebook-featured-profile.datebook-featured-compact .datebook-profile-comment-caption {';
				$css .= 'border-radius: calc(' . $css_data['featured_round_other'] . 'rem - 10%);';
				$css .= '}';

				$css .= '.datebook-other-container .datebook-featured-profile.datebook-featured-compact .datebook-profile-pictures-caption {';
				$css .= 'border-radius: calc(' . $css_data['featured_round_other'] . 'rem - 1%);';
				$css .= '}';

				$css .= '.datebook-other-container .datebook-featured-profile.datebook-featured-full img {';
				$css .= 'border-radius: calc(' . $css_data['featured_round_other'] . 'rem - 1%); border-bottom-left-radius: 0; border-bottom-right-radius: 0;';
				$css .= '}';

				$css .= '.datebook-other-container .datebook-featured-profile.datebook-featured-compact img {';
				$css .= 'border-radius: calc(' . $css_data['featured_round_other'] . 'rem - 1%);';
				$css .= '}';

				$css .= '.datebook-other-container .card.datebook-featured-profile.datebook-featured-full .card-body {';
				$css .= 'border-radius: calc(' . $css_data['featured_round_other'] . 'rem - 1%); border-top-left-radius: 0; border-top-right-radius: 0;';
				$css .= '}';
			}

			if( !empty($css_data['featured_shadow_other']) ) {

				$css .= '.datebook-other-container .datebook-featured-profile {';
				
				$css .= $this->shadows_list($css_data['featured_shadow_other']);

				$css .= '}';

			}
			
			
			// Top profiles on Home page
			if( !empty($css_data['top_shadow']) ) {

				$css .= '.datebook-front-container .datebook-top-shadow {';
				
				$css .= $this->shadows_list($css_data['top_shadow']);

				$css .= '}';

			}

			// Top profile padding body
			if( isset($css_data['top_padding']) ) {

				$css .= '.datebook-top-front .datebook-top-profile .card-body {';
				
				$css .= 'padding: ' . $css_data['top_padding'] . 'rem;';

				$css .= '}';

			}
			

			// Normal profiles on Home page
			if( !empty($css_data['normal_shadow_front']) ) {

				$css .= '.datebook-front-container .datebook-normal-profile img {';
				
				$css .= $this->shadows_list($css_data['normal_shadow_front']);

				$css .= '}';

			}


			// Chat sidebar: font
			if( !empty($css_data['chat_sidebar_font']) ) {
				$css .= '.datebook-panel-messages .datebook-chatters-sidebar .swiper-slide a.datebook-chat-sidebar-color, .datebook-panel-messages .datebook-chatters-sidebar .swiper-slide a.datebook-chat-sidebar-color:hover, .datebook-panel-messages .datebook-chatters-sidebar .swiper-slide button, .datebook-panel-messages .datebook-chatters-sidebar .swiper-slide button:hover {';
				if( !empty($css_data['chat_sidebar_font']['face']) ) {
					if ( array_key_exists( $css_data['chat_sidebar_font']['face'], $recognized_fonts ) ) {
						$css .= 'font-family: ' . $recognized_fonts[$css_data['chat_sidebar_font']['face']] . ';';
					}
				}
				if( !empty($css_data['chat_sidebar_font']['style']) ) {
					$css .= 'font-style: ' . $css_data['chat_sidebar_font']['style'] . ';';
				}
				if( !empty($css_data['chat_sidebar_font']['color']) ) {
					$css .= 'color: ' . $css_data['chat_sidebar_font']['color'] . ';';
				}
				$css .= '}';

				$css .= '.datebook-panel-messages .datebook-chatters-sidebar .swiper-slide a, .datebook-panel-messages .datebook-chatters-sidebar .swiper-slide a:hover, .datebook-panel-messages .datebook-chatters-sidebar .swiper-slide button, .datebook-panel-messages .datebook-chatters-sidebar .swiper-slide button:hover {';
				if( !empty($css_data['chat_sidebar_font']['size']) ) {
					$css .= 'font-size: ' . $css_data['chat_sidebar_font']['size'] . $css_data['chat_sidebar_font']['measure'] . ';';
				}
				$css .= '}';
			}
			

			// Chat sidebar: width
			if( !empty($css_data['chat_text_field_width']) ) {
				$css .= '.datebook-panel-messages .datebook-chat-container .list-group .datebook-chat-text-width, .datebook-panel-messages .datebook-chat-container .list-group .datebook-lastmessagenotread {';
				if($css_data['chat_text_field_width'] < 100 ) {
					$css .= 'width: auto;';
				}
				$css .= 'max-width: ' . $css_data['chat_text_field_width'] . '%;';
				$css .= '}';
			}
			// Chat sidebar: width
			if( !empty($css_data['chat_text_border_round']) ) {
				$css .= '.datebook-panel-messages .datebook-chat-container .list-group .datebook-chat-text-width {';
				$css .= 'border-radius: ' . $css_data['chat_text_border_round'] . 'rem;';
				$css .= '}';
				$css .= '.datebook-panel-messages .datebook-chat-container .list-group .datebook-lastmessagenotread, .datebook-panel-messages .datebook-chat-container .list-group #datebook-loader span {';
				$css .= 'border-radius: calc(' . $css_data['chat_text_border_round'] . 'rem - 1%);';
				$css .= '}';
			}

			// Chat popup notification: background color
			if( !empty($css_data['popup_notification_bg_color'])) {
				$css .= '.datebook-popup-design, .datebook-popup-design .datebook-popup-close {';
				$css .= 'background: #ffc107;';
				$css .= 'background: -webkit-' . $css_data['popup_notification_bg_color'] . ';';
				$css .= 'background: -ms-' . $css_data['popup_notification_bg_color'] . ';';
				$css .= 'background: -moz-' . $css_data['popup_notification_bg_color'] . ';';
				$css .= 'background: -o-' . $css_data['popup_notification_bg_color'] . ';';
				$css .= 'background: -webkit-' . $css_data['popup_notification_bg_color'] . ';';
				$css .= 'background: ' . $css_data['popup_notification_bg_color'] . ';';
				$css .= '}';
			}


		return $css;

	}


	/**
	 * Create empty transparent PNG image file
	 */
	function create_empty_png($image_name, $width, $height, $images_dir){
		$img = imagecreatetruecolor($width, $height);
		imagesavealpha($img, true);
		$color = imagecolorallocatealpha($img, 0, 0, 0, 127);
		imagefill($img, 0, 0, $color);

		$new_image = $images_dir . '/' . $image_name . '.png';

		imagepng($img, $new_image);

	}


	/**
	 * Create and save different No image files
	 */
	function save_no_image_files($css_data, $images_dir){

		global $datebook_image_sizes;

		//if ($css_data['datebook_custom_images']) {

			$image_dimensions = array();

			foreach($datebook_image_sizes as $key => $value) {
				$image_dimensions[$key] = $css_data[$key] ? explode(',', $css_data[$key]) : array($value['width'],$value['height']);
			}

			foreach($image_dimensions as $image_name => $item) {
				self::create_empty_png($image_name, $image_dimensions[$image_name][0], $image_dimensions[$image_name][1], $images_dir);
			}
		//}
	}


	/**
	 * Save some Style and Typography to file in wp-content/uploads/datebook-options
	 */
	function save_to_css_file($css_data) {

		// set up cache folder
		$upload_dir = wp_upload_dir();

		$cache_dir  = trailingslashit( $upload_dir['basedir'] . '/' . DATEBOOK_OPTIONS_FOLDER);
		$cache_url  = trailingslashit( $upload_dir['baseurl'] . '/' . DATEBOOK_OPTIONS_FOLDER);

		// Check and create cachedir
		if ( ! file_exists( $cache_dir ) ) {
			wp_mkdir_p( $cache_dir );
		}

		$images_dir  = trailingslashit( $upload_dir['basedir'] . '/' . DATEBOOK_OPTIONS_FOLDER . '/images');

		// Check and create images_dir
		if ( ! file_exists( $images_dir ) ) {
			wp_mkdir_p( $images_dir );
		}

		// Check if cache folder is writable
		if( is_writable( $cache_dir ) ) {

			//$has_cached = get_option( 'datebook-css-cached' );

			//if( ! $has_cached ) {

				// Generate cache css file				
				$css_file = $cache_dir . '/' . DATEBOOK_CUSTOM_STYLE . '.css';

				$css_body  = "/**\n";
				$css_body .= " * This file is generated by DateBook\n";
				$css_body .= " * Time created: " . date( 'd F Y, H:i:s' ) . "\n";
				$css_body .= " */\n\n\n";
				$css_body .= self::generate_css_data($css_data);

				if( ! function_exists( 'WP_Filesystem' ) ) {
					require_once( ABSPATH . 'wp-admin/includes/file.php' );
				}

				WP_Filesystem();

				global $wp_filesystem;

				if ( ! $wp_filesystem->put_contents( $css_file, $css_body, FS_CHMOD_FILE ) ) {
					update_option( 'datebook-css-cached', false );
				} else {
					update_option( 'datebook-css-cached', true );
				}


				// Create thumbnail for No image
				self::save_no_image_files($css_data, $images_dir);

			//}

		} else {

			// echo generated css directly if cache folder is not writable and in customizer
			add_action( 'wp_head', function(){  echo '<style>'. self::generate_css_data($css_data) .'</style>'; }, 99 );

		}

	}

	
	
	/**
	 * Display message when options have been saved
	 */

	function save_options_notice() {
		add_settings_error( 'datebook-options-framework', 'save_options', esc_html__( 'Theme Options have been saved', 'datebook' ), 'updated fade' );
	}


	/**
	 * Display message when designs have been saved
	 */

	function save_designs_notice() {
		add_settings_error( 'datebook-options-design', 'save_designs', esc_html__( 'Design options have been saved', 'datebook' ), 'updated fade' );
	}


	/**
	 * Get the default values for all the plugin options
	 *
	 * Get an array of all default values as set in
	 * options.php. The 'id','std' and 'type' keys need
	 * to be defined in the configuration array. In the
	 * event that these keys are not present the option
	 * will not be included in this function's output.
	 *
	 * @return array Re-keyed options configuration array.
	 *
	 */
	function get_default_values($frame_section) {

		$output = array();

		if ( $frame_section == 'options') {
			$config = & Options_Framework::_optionsframework_options('options');
		} else if ( $frame_section == 'design') {
			$config = & Options_Framework::_optionsframework_options('design');
		}

		foreach ( (array) $config as $option ) {
			if ( ! isset( $option['id'] ) ) {
				continue;
			}
			if ( ! isset( $option['std'] ) ) {
				continue;
			}
			if ( ! isset( $option['type'] ) ) {
				continue;
			}
			if ( has_filter( 'of_sanitize_' . $option['type'] ) ) {
				$output[$option['id']] = apply_filters( 'of_sanitize_' . $option['type'], $option['std'], $option );
			}
		}
		return $output;
	}


	/**
	 * Add options menu item to admin bar
	 */

	function optionsframework_admin_bar() {

		$menu = $this->menu_settings();

		global $wp_admin_bar;

		if ( 'menu' == $menu['mode'] ) {
			$href = admin_url( 'admin.php?page=' . $menu['menu_slug'] );
		} else {
			$href = admin_url( 'themes.php?page=' . $menu['menu_slug'] );
		}

		$args = array(
			'parent' => 'appearance',
			'id' => 'of_theme_options',
			'title' => $menu['menu_title'],
			'href' => $href
		);

		$wp_admin_bar->add_menu( apply_filters( 'optionsframework_admin_bar', $args ) );
	}

}