<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Verifies whether the current page communicates through HTTPS
 *
 * @return bool
 *
 */
function datebook_is_https() {

    $is_secure = false;

    if ( isset( $_SERVER['HTTPS'] ) && $_SERVER['HTTPS'] == 'on' ) {

        $is_secure = true;

    } elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_PROTO'] ) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https' || ! empty( $_SERVER['HTTP_X_FORWARDED_SSL'] ) && $_SERVER['HTTP_X_FORWARDED_SSL'] == 'on' ) {

        $is_secure = true;

    }

    return $is_secure;

}


/**
 * Function that returns only the date part of a date-time format
 *
 * @param string $date
 *
 * @return string
 *
 */
function datebook_sanitize_date( $date ) {

    if( !isset( $date ) )
        return;

    $date_time = explode( ' ', $date );

    return $date_time[0];

}


/**
 * Returns the url of the current page
 *
 * @param bool $strip_query_args - whether to eliminate query arguments from the url or not
 *
 * @return string
 *
 */
function datebook_get_current_page_url( $strip_query_args = false ) {

    $page_url = 'http';

    if ((isset($_SERVER["HTTPS"])) && ($_SERVER["HTTPS"] == "on"))
        $page_url .= "s";

    $page_url .= "://";

    if ($_SERVER["SERVER_PORT"] != "80")
        $page_url .= $_SERVER["SERVER_NAME"] . ":" . $_SERVER["SERVER_PORT"] . $_SERVER["REQUEST_URI"];
    else
        $page_url .= $_SERVER["SERVER_NAME"] . $_SERVER["REQUEST_URI"];


    // Remove query arguments
    if( $strip_query_args ) {
        $page_url_parts = explode( '?', $page_url );

        $page_url = $page_url_parts[0];

        // Keep query args "p" and "page_id" for non-beautified permalinks
        if( isset( $page_url_parts[1] ) ) {
            $page_url_query_args = explode( '&', $page_url_parts[1] );

            if( !empty( $page_url_query_args ) ) {
                foreach( $page_url_query_args as $key => $query_arg ) {

                    if( strpos( $query_arg, 'p=' ) === 0 ) {
                        $query_arg_parts = explode( '=', $query_arg );
                        $query_arg       = $query_arg_parts[0];
                        $query_arg_val   = $query_arg_parts[1];

                        $page_url = add_query_arg( array( $query_arg => $query_arg_val ), $page_url );
                    }

                    if( strpos( $query_arg, 'page_id=' ) === 0 ) {
                        $query_arg_parts = explode( '=', $query_arg );
                        $query_arg       = $query_arg_parts[0];
                        $query_arg_val   = $query_arg_parts[1];

                        $page_url = add_query_arg( array( $query_arg => $query_arg_val ), $page_url );
                    }

                }
            }
        }

    }


    /**
     * Filter the page url just before returning
     *
     * @param string $page_url
     *
     */
    $page_url = apply_filters( 'datebook_get_current_page_url', $page_url );

    return $page_url;

}


/**
 * Checks if there is a need to add the http:// prefix to a link and adds it. Returns the correct link.
 *
 * @param string $link
 *
 * @return string
 *
 */
function datebook_add_missing_http( $link = '' ) {

    $http = '';

    if ( preg_match( '#^(?:[a-z\d]+(?:-+[a-z\d]+)*\.)+[a-z]+(?::\d+)?(?:/|$)#i', $link ) ) { //if missing http(s)

        $http = 'http';
        if ((isset($_SERVER["HTTPS"])) && ($_SERVER["HTTPS"] == "on"))
            $http .= "s";
        $http .= "://";
    }

    return $http . $link;

}


/**
 * Function that return the IP address of the user. Checks for IPs (in order) in: 'HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR'
 *
 * @return string
 *
 */
function datebook_get_user_ip_address() {

    $ip_address = '';

    foreach (array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR') as $key){
        if (array_key_exists($key, $_SERVER) === true) {
            foreach ( array_map('trim', explode( ',', $_SERVER[$key]) ) as $ip ) {
                if ( filter_var($ip, FILTER_VALIDATE_IP) !== false ) {
                    return $ip;
                }
            }
        }
    }

    return $ip_address;
    
}


/**
 * Sanitizes the values of an array recursivelly
 *
 * @param array $array
 *
 * @return array
 *
 */
function datebook_array_sanitize_text_field( $array = array() ) {

    if( empty( $array ) || ! is_array( $array ) )
        return array();

    foreach( $array as $key => $value ) {

        if( is_array( $value ) )
            $array[$key] = datebook_array_sanitize_text_field( $value );

        else
            $array[$key] = sanitize_text_field( $value );

    }

    return $array;

}


/**
 * Removes the script tags from the values of an array recursivelly
 *
 * @param array $array
 *
 * @return array
 *
 */
function datebook_array_strip_script_tags( $array = array() ) {

    if( empty( $array ) || ! is_array( $array ) )
        return array();
		
    foreach( $array as $key => $value ) {

        if( is_array( $value ) ){
			$array[$key] = datebook_array_strip_script_tags( $value );
		} else {
			$array[$key] = wp_strip_all_tags( $value );
		}

    }

    return $array;

}


/**
 * Callback for the "wp_kses_allowed_html" filter to add iframes to the allowed tags
 *
 * @param array  $tags
 * @param strint $context
 *
 * @return array
 *
 */
function datebook_wp_kses_allowed_html_iframe( $tags = array(), $context = '' ) {
    
    if ( 'post' === $context ) {

        $tags['iframe'] = array(
            'src'             => true,
            'height'          => true,
            'width'           => true,
            'frameborder'     => true,
            'allowfullscreen' => true,
        );

    }

    return $tags;

}


/**
 * Copy of WordPress's default _deprecated_function() function, which is marked as private 
 *
 */
function _datebook_deprecated_function( $function, $version, $replacement = null ) {
 
    /**
     * Filters whether to trigger an error for deprecated functions.
     *
     * @param bool $trigger Whether to trigger the error for deprecated functions. Default true.
     *
     */
    if ( WP_DEBUG && apply_filters( 'datebook_deprecated_function_trigger_error', true ) ) {
        if ( function_exists( '__' ) ) {
            if ( ! is_null( $replacement ) ) {
                /* translators: 1: PHP function name, 2: version number, 3: alternative function name */
                trigger_error( sprintf( esc_html__('%1$s is <strong>deprecated</strong> since version %2$s! Use %3$s instead.'), $function, $version, $replacement ) );
            } else {
                /* translators: 1: PHP function name, 2: version number */
                trigger_error( sprintf( esc_html__('%1$s is <strong>deprecated</strong> since version %2$s with no alternative available.'), $function, $version ) );
            }
        } else {
            if ( ! is_null( $replacement ) ) {
                trigger_error( sprintf( '%1$s is <strong>deprecated</strong> since version %2$s! Use %3$s instead.', $function, $version, $replacement ) );
            } else {
                trigger_error( sprintf( '%1$s is <strong>deprecated</strong> since version %2$s with no alternative available.', $function, $version ) );
            }
        }
    }
}










    /**
     * Function that is a wrapper for default WP function get_post_meta,
     * but if provided only the $post_id will return an associative array with values,
     * not an associative array of array
     *
     * @param $post_id      - the ID of the post
     * @param $key          - the post meta key
     * @param $single
     *
     */
    function datebook_get_post_meta( $post_id, $key = '', $single = false ) {

        if( empty( $key ) ) {
            $post_meta = get_post_meta( $post_id );

            foreach( $post_meta as $key => $value ) {
                $post_meta[$key] = $value[0];
            }

            return $post_meta;
        }

        return get_post_meta( $post_id, $key, $single );

    }


    /**
     * Function that returns all the users that are not members yet
     *
     * @return array
     *
     */
    function datebook_get_users_non_members( $args = array() ) {

        global $wpdb;
		
		$table_name = strtolower($wpdb->prefix) . 'datebook_member_subscriptions';

        $defaults = array(
            'orderby' => 'ID',
            'offset'  => '',
            'limit'   => ''
        );

        $args = apply_filters( 'datebook_get_users_non_members_args', wp_parse_args( $args, $defaults ), $args, $defaults );

        // Start query string
        $query_string = "SELECT {$wpdb->users}.ID ";

        // Query string sections
        $query_from   = "FROM {$wpdb->users} ";
        $query_join   = "LEFT JOIN {$table_name} ON {$wpdb->users}.ID = {$table_name}.user_id ";
        $query_where  = "WHERE {$table_name}.user_id is null ";

        $query_limit = '';
        if( !empty( $args['limit'] ) )
            $query_limit = "LIMIT " . $args['limit'] . " ";

        $query_offset = '';
        if( !empty( $args['offset'] ) )
            $query_offset = "OFFSET " . $args['offset'] . " ";


        // Concatenate the sections into the full query string
        $query_string .= $query_from . $query_join . $query_where . $query_limit . $query_offset;

        $results = $wpdb->get_results( $query_string, ARRAY_A );

        $users = array();

        if( !empty( $results ) ) {
            foreach( $results as $result ) {
                $users[] = new WP_User( $result['ID'] );
            }
        }

        return apply_filters( 'datebook_get_users_non_members', $users, $args );

    }


    /**
     * Handles errors in the front end
     *
     */
    function datebook_errors() {
        static $wp_errors;

        return ( isset($wp_errors) ? $wp_errors : ( $wp_errors = new WP_Error( null, null, null ) ) );
    }


    /**
     * Handles success messages in front end
     *
     */
    function datebook_success() {
        static $datebook_success;

        return ( isset($datebook_success) ? $datebook_success : ( $datebook_success = new DateBook_Success( null, null ) ) );
    }


    /**
     * Displays general errors at the top of the forms
     *
     */
    function datebook_display_errors_form_general() {

        $form_error = datebook_errors()->get_error_message( 'form_general' );

        if( empty( $form_error ) )
            return;

        echo '<div class="datebook-form-errors-wrapper">';
        echo '<p>' . $form_error . '</p>';
        echo '</div>';

    }
    add_action( 'datebook_register_form_top', 'datebook_display_errors_form_general' );
    add_action( 'datebook_new_subscription_form_top', 'datebook_display_errors_form_general' );
    add_action( 'datebook_upgrade_subscription_form_top', 'datebook_display_errors_form_general' );
    add_action( 'datebook_renew_subscription_form_bottom', 'datebook_display_errors_form_general' );
    add_action( 'datebook_retry_payment_form_top', 'datebook_display_errors_form_general' );


    /**
     * Adds a hidden span with the text placeholder for the submit buttons while
     * processing the data
     *
     */
    function datebook_add_hidden_submit_button_loading_placeholder_text() {

        echo '<span id="datebook-submit-button-loading-placeholder-text" style="display: none;">' . apply_filters( 'datebook_submit_button_loading_placeholder_text', esc_html__( 'Processing. Please wait...', 'datebook' ) ) . '</span>';

    }
    add_action( 'datebook_register_form_bottom', 'datebook_add_hidden_submit_button_loading_placeholder_text' );
    add_action( 'datebook_new_subscription_form_bottom', 'datebook_add_hidden_submit_button_loading_placeholder_text' );
    add_action( 'datebook_upgrade_subscription_form_bottom', 'datebook_add_hidden_submit_button_loading_placeholder_text' );
    add_action( 'datebook_renew_subscription_form_bottom', 'datebook_add_hidden_submit_button_loading_placeholder_text' );
    add_action( 'datebook_retry_payment_form_bottom', 'datebook_add_hidden_submit_button_loading_placeholder_text' );


    /**
     * Function that echoes the errors of a field
     *
     * @param array $field_errors - an array containing the errors
     *
     */
    function datebook_display_field_errors( $field_errors = array(), $return = false ) {

        $output = '';

        if( !empty( $field_errors ) ) {
            $output = '<div class="datebook_field-errors-wrapper">';

            foreach( $field_errors as $field_error ) {
                $output .= '<p>' . $field_error . '</p>';
            }

            $output .= '</div>';
        }

        if( $return )
            return $output;
        else
            echo $output;

    }


    /**
     * Function that echoes success messages
     *
     * @param array $messages - an array containing the messages
     *
     */
    function datebook_display_success_messages( $messages = array(), $return = false ) {

        $output = '';

        if( !empty( $messages ) ) {
            $output = '<div class="datebook_success-messages-wrapper">';

            foreach( $messages as $message ) {
                $output .= '<p>' . $message . '</p>';
            }

            $output .= '</div>';
        }

        if( $return )
            return $output;
        else
            echo $output;

    }


    /**
     * Returns an array with the currency codes and their names
     *
     * @return array
     *
     */
    function datebook_get_currencies() {

        $currencies = array(
            'USD'   => esc_html__( 'US Dollar', 'datebook' ),
            'EUR'   => esc_html__( 'Euro', 'datebook' ),
            'GBP'   => esc_html__( 'Pound Sterling', 'datebook' ),
            'CAD'   => esc_html__( 'Canadian Dollar', 'datebook' ),
            'AUD'   => esc_html__( 'Australian Dollar', 'datebook' ),
            'BRL'   => esc_html__( 'Brazilian Real', 'datebook' ),
            'CZK'   => esc_html__( 'Czech Koruna', 'datebook' ),
            'DKK'   => esc_html__( 'Danish Krone', 'datebook' ),
            'HKD'   => esc_html__( 'Hong Kong Dollar', 'datebook' ),
            'HUF'   => esc_html__( 'Hungarian Forint', 'datebook' ),
            'ILS'   => esc_html__( 'Israeli New Sheqel', 'datebook' ),
            'JPY'   => esc_html__( 'Japanese Yen', 'datebook' ),
            'MYR'   => esc_html__( 'Malaysian Ringgit', 'datebook' ),
            'MXN'   => esc_html__( 'Mexican Peso', 'datebook' ),
            'NOK'   => esc_html__( 'Norwegian Krone', 'datebook' ),
            'NZD'   => esc_html__( 'New Zealand Dollar', 'datebook' ),
            'PHP'   => esc_html__( 'Philippine Peso', 'datebook' ),
            'PLN'   => esc_html__( 'Polish Zloty', 'datebook' ),
            'RUB'   => esc_html__( 'Russian Ruble', 'datebook' ),
            'SGD'   => esc_html__( 'Singapore Dollar', 'datebook' ),
            'SEK'   => esc_html__( 'Swedish Krona', 'datebook' ),
            'CHF'   => esc_html__( 'Swiss Franc', 'datebook' ),
            'TWD'   => esc_html__( 'Taiwan New Dollar', 'datebook' ),
            'THB'   => esc_html__( 'Thai Baht', 'datebook' ),
            'TRY'   => esc_html__( 'Turkish Lira', 'datebook' )
        );

        return apply_filters( 'datebook_currencies', $currencies );

    }


    /**
     * Given a currency code returns a string with the currency symbol as HTML entity
     *
     * @param string $currency_code
     *
     * @return string
     *
     */
    function datebook_get_currency_symbol( $currency_code ) {

        $currencies = apply_filters('datebook_currency_symbols',
            array(
                'AED' => '&#1583;.&#1573;', // ?
                'AFN' => '&#65;&#102;',
                'ALL' => '&#76;&#101;&#107;',
                'AMD' => '',
                'ANG' => '&#402;',
                'AOA' => '&#75;&#122;', // ?
                'ARS' => '&#36;',
                'AUD' => '&#36;',
                'AWG' => '&#402;',
                'AZN' => '&#1084;&#1072;&#1085;',
                'BAM' => '&#75;&#77;',
                'BBD' => '&#36;',
                'BDT' => '&#2547;', // ?
                'BGN' => '&#1083;&#1074;',
                'BHD' => '.&#1583;.&#1576;', // ?
                'BIF' => '&#70;&#66;&#117;', // ?
                'BMD' => '&#36;',
                'BND' => '&#36;',
                'BOB' => '&#36;&#98;',
                'BRL' => '&#82;&#36;',
                'BSD' => '&#36;',
                'BTN' => '&#78;&#117;&#46;', // ?
                'BWP' => '&#80;',
                'BYR' => '&#112;&#46;',
                'BZD' => '&#66;&#90;&#36;',
                'CAD' => '&#36;',
                'CDF' => '&#70;&#67;',
                'CHF' => '&#67;&#72;&#70;',
                'CLF' => '', // ?
                'CLP' => '&#36;',
                'CNY' => '&#165;',
                'COP' => '&#36;',
                'CRC' => '&#8353;',
                'CUP' => '&#8396;',
                'CVE' => '&#36;', // ?
                'CZK' => '&#75;&#269;',
                'DJF' => '&#70;&#100;&#106;', // ?
                'DKK' => '&#107;&#114;',
                'DOP' => '&#82;&#68;&#36;',
                'DZD' => '&#1583;&#1580;', // ?
                'EGP' => '&#163;',
                'ETB' => '&#66;&#114;',
                'EUR' => '&#8364;',
                'FJD' => '&#36;',
                'FKP' => '&#163;',
                'GBP' => '&#163;',
                'GEL' => '&#4314;', // ?
                'GHS' => '&#162;',
                'GIP' => '&#163;',
                'GMD' => '&#68;', // ?
                'GNF' => '&#70;&#71;', // ?
                'GTQ' => '&#81;',
                'GYD' => '&#36;',
                'HKD' => '&#36;',
                'HNL' => '&#76;',
                'HRK' => '&#107;&#110;',
                'HTG' => '&#71;', // ?
                'HUF' => '&#70;&#116;',
                'IDR' => '&#82;&#112;',
                'ILS' => '&#8362;',
                'INR' => '&#8377;',
                'IQD' => '&#1593;.&#1583;', // ?
                'IRR' => '&#65020;',
                'ISK' => '&#107;&#114;',
                'JEP' => '&#163;',
                'JMD' => '&#74;&#36;',
                'JOD' => '&#74;&#68;', // ?
                'JPY' => '&#165;',
                'KES' => '&#75;&#83;&#104;', // ?
                'KGS' => '&#1083;&#1074;',
                'KHR' => '&#6107;',
                'KMF' => '&#67;&#70;', // ?
                'KPW' => '&#8361;',
                'KRW' => '&#8361;',
                'KWD' => '&#1583;.&#1603;', // ?
                'KYD' => '&#36;',
                'KZT' => '&#1083;&#1074;',
                'LAK' => '&#8365;',
                'LBP' => '&#163;',
                'LKR' => '&#8360;',
                'LRD' => '&#36;',
                'LSL' => '&#76;', // ?
                'LTL' => '&#76;&#116;',
                'LVL' => '&#76;&#115;',
                'LYD' => '&#1604;.&#1583;', // ?
                'MAD' => '&#1583;.&#1605;.', //?
                'MDL' => '&#76;',
                'MGA' => '&#65;&#114;', // ?
                'MKD' => '&#1076;&#1077;&#1085;',
                'MMK' => '&#75;',
                'MNT' => '&#8366;',
                'MOP' => '&#77;&#79;&#80;&#36;', // ?
                'MRO' => '&#85;&#77;', // ?
                'MUR' => '&#8360;', // ?
                'MVR' => '.&#1923;', // ?
                'MWK' => '&#77;&#75;',
                'MXN' => '&#36;',
                'MYR' => '&#82;&#77;',
                'MZN' => '&#77;&#84;',
                'NAD' => '&#36;',
                'NGN' => '&#8358;',
                'NIO' => '&#67;&#36;',
                'NOK' => '&#107;&#114;',
                'NPR' => '&#8360;',
                'NZD' => '&#36;',
                'OMR' => '&#65020;',
                'PAB' => '&#66;&#47;&#46;',
                'PEN' => '&#83;&#47;&#46;',
                'PGK' => '&#75;', // ?
                'PHP' => '&#8369;',
                'PKR' => '&#8360;',
                'PLN' => '&#122;&#322;',
                'PYG' => '&#71;&#115;',
                'QAR' => '&#65020;',
                'RON' => '&#108;&#101;&#105;',
                'RSD' => '&#1044;&#1080;&#1085;&#46;',
                'RUB' => '&#1088;&#1091;&#1073;',
                'RWF' => '&#1585;.&#1587;',
                'SAR' => '&#65020;',
                'SBD' => '&#36;',
                'SCR' => '&#8360;',
                'SDG' => '&#163;', // ?
                'SEK' => '&#107;&#114;',
                'SGD' => '&#36;',
                'SHP' => '&#163;',
                'SLL' => '&#76;&#101;', // ?
                'SOS' => '&#83;',
                'SRD' => '&#36;',
                'STD' => '&#68;&#98;', // ?
                'SVC' => '&#36;',
                'SYP' => '&#163;',
                'SZL' => '&#76;', // ?
                'THB' => '&#3647;',
                'TJS' => '&#84;&#74;&#83;', // ? TJS (guess)
                'TMT' => '&#109;',
                'TND' => '&#1583;.&#1578;',
                'TOP' => '&#84;&#36;',
                'TRY' => '&#8356;', // New Turkey Lira (old symbol used)
                'TTD' => '&#36;',
                'TWD' => '&#78;&#84;&#36;',
                'TZS' => '',
                'UAH' => '&#8372;',
                'UGX' => '&#85;&#83;&#104;',
                'USD' => '&#36;',
                'UYU' => '&#36;&#85;',
                'UZS' => '&#1083;&#1074;',
                'VEF' => '&#66;&#115;',
                'VND' => '&#8363;',
                'VUV' => '&#86;&#84;',
                'WST' => '&#87;&#83;&#36;',
                'XAF' => '&#70;&#67;&#70;&#65;',
                'XCD' => '&#36;',
                'XDR' => '',
                'XOF' => '',
                'XPF' => '&#70;',
                'YER' => '&#65020;',
                'ZAR' => '&#82;',
                'ZMK' => '&#90;&#75;', // ?
                'ZWL' => '&#90;&#36;',
            )
        );

        $currency_symbol = ( isset( $currencies[$currency_code] ) ? $currencies[$currency_code] : $currency_code );

        return $currency_symbol;

    }


    /**
     * Function that returns an array with countries using country codes as keys
     *
     * @return array
     *
     */
    function datebook_get_countries() {

        $country_array = apply_filters( 'datebook_get_countries',
            array(
                ''	 => '',
                'AF' => esc_html__( 'Afghanistan', 'datebook' ),
                'AX' => esc_html__( 'Aland Islands', 'datebook' ),
                'AL' => esc_html__( 'Albania', 'datebook' ),
                'DZ' => esc_html__( 'Algeria', 'datebook' ),
                'AS' => esc_html__( 'American Samoa', 'datebook' ),
                'AD' => esc_html__( 'Andorra', 'datebook' ),
                'AO' => esc_html__( 'Angola', 'datebook' ),
                'AI' => esc_html__( 'Anguilla', 'datebook' ),
                'AQ' => esc_html__( 'Antarctica', 'datebook' ),
                'AG' => esc_html__( 'Antigua and Barbuda', 'datebook' ),
                'AR' => esc_html__( 'Argentina', 'datebook' ),
                'AM' => esc_html__( 'Armenia', 'datebook' ),
                'AW' => esc_html__( 'Aruba', 'datebook' ),
                'AU' => esc_html__( 'Australia', 'datebook' ),
                'AT' => esc_html__( 'Austria', 'datebook' ),
                'AZ' => esc_html__( 'Azerbaijan', 'datebook' ),
                'BS' => esc_html__( 'Bahamas', 'datebook' ),
                'BH' => esc_html__( 'Bahrain', 'datebook' ),
                'BD' => esc_html__( 'Bangladesh', 'datebook' ),
                'BB' => esc_html__( 'Barbados', 'datebook' ),
                'BY' => esc_html__( 'Belarus', 'datebook' ),
                'BE' => esc_html__( 'Belgium', 'datebook' ),
                'BZ' => esc_html__( 'Belize', 'datebook' ),
                'BJ' => esc_html__( 'Benin', 'datebook' ),
                'BM' => esc_html__( 'Bermuda', 'datebook' ),
                'BT' => esc_html__( 'Bhutan', 'datebook' ),
                'BO' => esc_html__( 'Bolivia', 'datebook' ),
                'BQ' => esc_html__( 'Bonaire, Saint Eustatius and Saba', 'datebook' ),
                'BA' => esc_html__( 'Bosnia and Herzegovina', 'datebook' ),
                'BW' => esc_html__( 'Botswana', 'datebook' ),
                'BV' => esc_html__( 'Bouvet Island', 'datebook' ),
                'BR' => esc_html__( 'Brazil', 'datebook' ),
                'IO' => esc_html__( 'British Indian Ocean Territory', 'datebook' ),
                'VG' => esc_html__( 'British Virgin Islands', 'datebook' ),
                'BN' => esc_html__( 'Brunei', 'datebook' ),
                'BG' => esc_html__( 'Bulgaria', 'datebook' ),
                'BF' => esc_html__( 'Burkina Faso', 'datebook' ),
                'BI' => esc_html__( 'Burundi', 'datebook' ),
                'KH' => esc_html__( 'Cambodia', 'datebook' ),
                'CM' => esc_html__( 'Cameroon', 'datebook' ),
                'CA' => esc_html__( 'Canada', 'datebook' ),
                'CV' => esc_html__( 'Cape Verde', 'datebook' ),
                'KY' => esc_html__( 'Cayman Islands', 'datebook' ),
                'CF' => esc_html__( 'Central African Republic', 'datebook' ),
                'TD' => esc_html__( 'Chad', 'datebook' ),
                'CL' => esc_html__( 'Chile', 'datebook' ),
                'CN' => esc_html__( 'China', 'datebook' ),
                'CX' => esc_html__( 'Christmas Island', 'datebook' ),
                'CC' => esc_html__( 'Cocos Islands', 'datebook' ),
                'CO' => esc_html__( 'Colombia', 'datebook' ),
                'KM' => esc_html__( 'Comoros', 'datebook' ),
                'CK' => esc_html__( 'Cook Islands', 'datebook' ),
                'CR' => esc_html__( 'Costa Rica', 'datebook' ),
                'HR' => esc_html__( 'Croatia', 'datebook' ),
                'CU' => esc_html__( 'Cuba', 'datebook' ),
                'CW' => esc_html__( 'Curacao', 'datebook' ),
                'CY' => esc_html__( 'Cyprus', 'datebook' ),
                'CZ' => esc_html__( 'Czech Republic', 'datebook' ),
                'CD' => esc_html__( 'Democratic Republic of the Congo', 'datebook' ),
                'DK' => esc_html__( 'Denmark', 'datebook' ),
                'DJ' => esc_html__( 'Djibouti', 'datebook' ),
                'DM' => esc_html__( 'Dominica', 'datebook' ),
                'DO' => esc_html__( 'Dominican Republic', 'datebook' ),
                'TL' => esc_html__( 'East Timor', 'datebook' ),
                'EC' => esc_html__( 'Ecuador', 'datebook' ),
                'EG' => esc_html__( 'Egypt', 'datebook' ),
                'SV' => esc_html__( 'El Salvador', 'datebook' ),
                'GQ' => esc_html__( 'Equatorial Guinea', 'datebook' ),
                'ER' => esc_html__( 'Eritrea', 'datebook' ),
                'EE' => esc_html__( 'Estonia', 'datebook' ),
                'ET' => esc_html__( 'Ethiopia', 'datebook' ),
                'FK' => esc_html__( 'Falkland Islands', 'datebook' ),
                'FO' => esc_html__( 'Faroe Islands', 'datebook' ),
                'FJ' => esc_html__( 'Fiji', 'datebook' ),
                'FI' => esc_html__( 'Finland', 'datebook' ),
                'FR' => esc_html__( 'France', 'datebook' ),
                'GF' => esc_html__( 'French Guiana', 'datebook' ),
                'PF' => esc_html__( 'French Polynesia', 'datebook' ),
                'TF' => esc_html__( 'French Southern Territories', 'datebook' ),
                'GA' => esc_html__( 'Gabon', 'datebook' ),
                'GM' => esc_html__( 'Gambia', 'datebook' ),
                'GE' => esc_html__( 'Georgia', 'datebook' ),
                'DE' => esc_html__( 'Germany', 'datebook' ),
                'GH' => esc_html__( 'Ghana', 'datebook' ),
                'GI' => esc_html__( 'Gibraltar', 'datebook' ),
                'GR' => esc_html__( 'Greece', 'datebook' ),
                'GL' => esc_html__( 'Greenland', 'datebook' ),
                'GD' => esc_html__( 'Grenada', 'datebook' ),
                'GP' => esc_html__( 'Guadeloupe', 'datebook' ),
                'GU' => esc_html__( 'Guam', 'datebook' ),
                'GT' => esc_html__( 'Guatemala', 'datebook' ),
                'GG' => esc_html__( 'Guernsey', 'datebook' ),
                'GN' => esc_html__( 'Guinea', 'datebook' ),
                'GW' => esc_html__( 'Guinea-Bissau', 'datebook' ),
                'GY' => esc_html__( 'Guyana', 'datebook' ),
                'HT' => esc_html__( 'Haiti', 'datebook' ),
                'HM' => esc_html__( 'Heard Island and McDonald Islands', 'datebook' ),
                'HN' => esc_html__( 'Honduras', 'datebook' ),
                'HK' => esc_html__( 'Hong Kong', 'datebook' ),
                'HU' => esc_html__( 'Hungary', 'datebook' ),
                'IS' => esc_html__( 'Iceland', 'datebook' ),
                'IN' => esc_html__( 'India', 'datebook' ),
                'ID' => esc_html__( 'Indonesia', 'datebook' ),
                'IR' => esc_html__( 'Iran', 'datebook' ),
                'IQ' => esc_html__( 'Iraq', 'datebook' ),
                'IE' => esc_html__( 'Ireland', 'datebook' ),
                'IM' => esc_html__( 'Isle of Man', 'datebook' ),
                'IL' => esc_html__( 'Israel', 'datebook' ),
                'IT' => esc_html__( 'Italy', 'datebook' ),
                'CI' => esc_html__( 'Ivory Coast', 'datebook' ),
                'JM' => esc_html__( 'Jamaica', 'datebook' ),
                'JP' => esc_html__( 'Japan', 'datebook' ),
                'JE' => esc_html__( 'Jersey', 'datebook' ),
                'JO' => esc_html__( 'Jordan', 'datebook' ),
                'KZ' => esc_html__( 'Kazakhstan', 'datebook' ),
                'KE' => esc_html__( 'Kenya', 'datebook' ),
                'KI' => esc_html__( 'Kiribati', 'datebook' ),
                'XK' => esc_html__( 'Kosovo', 'datebook' ),
                'KW' => esc_html__( 'Kuwait', 'datebook' ),
                'KG' => esc_html__( 'Kyrgyzstan', 'datebook' ),
                'LA' => esc_html__( 'Laos', 'datebook' ),
                'LV' => esc_html__( 'Latvia', 'datebook' ),
                'LB' => esc_html__( 'Lebanon', 'datebook' ),
                'LS' => esc_html__( 'Lesotho', 'datebook' ),
                'LR' => esc_html__( 'Liberia', 'datebook' ),
                'LY' => esc_html__( 'Libya', 'datebook' ),
                'LI' => esc_html__( 'Liechtenstein', 'datebook' ),
                'LT' => esc_html__( 'Lithuania', 'datebook' ),
                'LU' => esc_html__( 'Luxembourg', 'datebook' ),
                'MO' => esc_html__( 'Macao', 'datebook' ),
                'MK' => esc_html__( 'Macedonia', 'datebook' ),
                'MG' => esc_html__( 'Madagascar', 'datebook' ),
                'MW' => esc_html__( 'Malawi', 'datebook' ),
                'MY' => esc_html__( 'Malaysia', 'datebook' ),
                'MV' => esc_html__( 'Maldives', 'datebook' ),
                'ML' => esc_html__( 'Mali', 'datebook' ),
                'MT' => esc_html__( 'Malta', 'datebook' ),
                'MH' => esc_html__( 'Marshall Islands', 'datebook' ),
                'MQ' => esc_html__( 'Martinique', 'datebook' ),
                'MR' => esc_html__( 'Mauritania', 'datebook' ),
                'MU' => esc_html__( 'Mauritius', 'datebook' ),
                'YT' => esc_html__( 'Mayotte', 'datebook' ),
                'MX' => esc_html__( 'Mexico', 'datebook' ),
                'FM' => esc_html__( 'Micronesia', 'datebook' ),
                'MD' => esc_html__( 'Moldova', 'datebook' ),
                'MC' => esc_html__( 'Monaco', 'datebook' ),
                'MN' => esc_html__( 'Mongolia', 'datebook' ),
                'ME' => esc_html__( 'Montenegro', 'datebook' ),
                'MS' => esc_html__( 'Montserrat', 'datebook' ),
                'MA' => esc_html__( 'Morocco', 'datebook' ),
                'MZ' => esc_html__( 'Mozambique', 'datebook' ),
                'MM' => esc_html__( 'Myanmar', 'datebook' ),
                'NA' => esc_html__( 'Namibia', 'datebook' ),
                'NR' => esc_html__( 'Nauru', 'datebook' ),
                'NP' => esc_html__( 'Nepal', 'datebook' ),
                'NL' => esc_html__( 'Netherlands', 'datebook' ),
                'NC' => esc_html__( 'New Caledonia', 'datebook' ),
                'NZ' => esc_html__( 'New Zealand', 'datebook' ),
                'NI' => esc_html__( 'Nicaragua', 'datebook' ),
                'NE' => esc_html__( 'Niger', 'datebook' ),
                'NG' => esc_html__( 'Nigeria', 'datebook' ),
                'NU' => esc_html__( 'Niue', 'datebook' ),
                'NF' => esc_html__( 'Norfolk Island', 'datebook' ),
                'KP' => esc_html__( 'North Korea', 'datebook' ),
                'MP' => esc_html__( 'Northern Mariana Islands', 'datebook' ),
                'NO' => esc_html__( 'Norway', 'datebook' ),
                'OM' => esc_html__( 'Oman', 'datebook' ),
                'PK' => esc_html__( 'Pakistan', 'datebook' ),
                'PW' => esc_html__( 'Palau', 'datebook' ),
                'PS' => esc_html__( 'Palestinian Territory', 'datebook' ),
                'PA' => esc_html__( 'Panama', 'datebook' ),
                'PG' => esc_html__( 'Papua New Guinea', 'datebook' ),
                'PY' => esc_html__( 'Paraguay', 'datebook' ),
                'PE' => esc_html__( 'Peru', 'datebook' ),
                'PH' => esc_html__( 'Philippines', 'datebook' ),
                'PN' => esc_html__( 'Pitcairn', 'datebook' ),
                'PL' => esc_html__( 'Poland', 'datebook' ),
                'PT' => esc_html__( 'Portugal', 'datebook' ),
                'PR' => esc_html__( 'Puerto Rico', 'datebook' ),
                'QA' => esc_html__( 'Qatar', 'datebook' ),
                'CG' => esc_html__( 'Republic of the Congo', 'datebook' ),
                'RE' => esc_html__( 'Reunion', 'datebook' ),
                'RO' => esc_html__( 'Romania', 'datebook' ),
                'RU' => esc_html__( 'Russia', 'datebook' ),
                'RW' => esc_html__( 'Rwanda', 'datebook' ),
                'BL' => esc_html__( 'Saint Barthelemy', 'datebook' ),
                'SH' => esc_html__( 'Saint Helena', 'datebook' ),
                'KN' => esc_html__( 'Saint Kitts and Nevis', 'datebook' ),
                'LC' => esc_html__( 'Saint Lucia', 'datebook' ),
                'MF' => esc_html__( 'Saint Martin', 'datebook' ),
                'PM' => esc_html__( 'Saint Pierre and Miquelon', 'datebook' ),
                'VC' => esc_html__( 'Saint Vincent and the Grenadines', 'datebook' ),
                'WS' => esc_html__( 'Samoa', 'datebook' ),
                'SM' => esc_html__( 'San Marino', 'datebook' ),
                'ST' => esc_html__( 'Sao Tome and Principe', 'datebook' ),
                'SA' => esc_html__( 'Saudi Arabia', 'datebook' ),
                'SN' => esc_html__( 'Senegal', 'datebook' ),
                'RS' => esc_html__( 'Serbia', 'datebook' ),
                'SC' => esc_html__( 'Seychelles', 'datebook' ),
                'SL' => esc_html__( 'Sierra Leone', 'datebook' ),
                'SG' => esc_html__( 'Singapore', 'datebook' ),
                'SX' => esc_html__( 'Sint Maarten', 'datebook' ),
                'SK' => esc_html__( 'Slovakia', 'datebook' ),
                'SI' => esc_html__( 'Slovenia', 'datebook' ),
                'SB' => esc_html__( 'Solomon Islands', 'datebook' ),
                'SO' => esc_html__( 'Somalia', 'datebook' ),
                'ZA' => esc_html__( 'South Africa', 'datebook' ),
                'GS' => esc_html__( 'South Georgia and the South Sandwich Islands', 'datebook' ),
                'KR' => esc_html__( 'South Korea', 'datebook' ),
                'SS' => esc_html__( 'South Sudan', 'datebook' ),
                'ES' => esc_html__( 'Spain', 'datebook' ),
                'LK' => esc_html__( 'Sri Lanka', 'datebook' ),
                'SD' => esc_html__( 'Sudan', 'datebook' ),
                'SR' => esc_html__( 'Suriname', 'datebook' ),
                'SJ' => esc_html__( 'Svalbard and Jan Mayen', 'datebook' ),
                'SZ' => esc_html__( 'Swaziland', 'datebook' ),
                'SE' => esc_html__( 'Sweden', 'datebook' ),
                'CH' => esc_html__( 'Switzerland', 'datebook' ),
                'SY' => esc_html__( 'Syria', 'datebook' ),
                'TW' => esc_html__( 'Taiwan', 'datebook' ),
                'TJ' => esc_html__( 'Tajikistan', 'datebook' ),
                'TZ' => esc_html__( 'Tanzania', 'datebook' ),
                'TH' => esc_html__( 'Thailand', 'datebook' ),
                'TG' => esc_html__( 'Togo', 'datebook' ),
                'TK' => esc_html__( 'Tokelau', 'datebook' ),
                'TO' => esc_html__( 'Tonga', 'datebook' ),
                'TT' => esc_html__( 'Trinidad and Tobago', 'datebook' ),
                'TN' => esc_html__( 'Tunisia', 'datebook' ),
                'TR' => esc_html__( 'Turkey', 'datebook' ),
                'TM' => esc_html__( 'Turkmenistan', 'datebook' ),
                'TC' => esc_html__( 'Turks and Caicos Islands', 'datebook' ),
                'TV' => esc_html__( 'Tuvalu', 'datebook' ),
                'VI' => esc_html__( 'U.S. Virgin Islands', 'datebook' ),
                'UG' => esc_html__( 'Uganda', 'datebook' ),
                'UA' => esc_html__( 'Ukraine', 'datebook' ),
                'AE' => esc_html__( 'United Arab Emirates', 'datebook' ),
                'GB' => esc_html__( 'United Kingdom', 'datebook' ),
                'US' => esc_html__( 'United States', 'datebook' ),
                'UM' => esc_html__( 'United States Minor Outlying Islands', 'datebook' ),
                'UY' => esc_html__( 'Uruguay', 'datebook' ),
                'UZ' => esc_html__( 'Uzbekistan', 'datebook' ),
                'VU' => esc_html__( 'Vanuatu', 'datebook' ),
                'VA' => esc_html__( 'Vatican', 'datebook' ),
                'VE' => esc_html__( 'Venezuela', 'datebook' ),
                'VN' => esc_html__( 'Vietnam', 'datebook' ),
                'WF' => esc_html__( 'Wallis and Futuna', 'datebook' ),
                'EH' => esc_html__( 'Western Sahara', 'datebook' ),
                'YE' => esc_html__( 'Yemen', 'datebook' ),
                'ZM' => esc_html__( 'Zambia', 'datebook' ),
                'ZW' => esc_html__( 'Zimbabwe', 'datebook' ),
            )
        );

        return $country_array;
    }



    /**
     * Function that returns the current user id or the current user that is edited in front-end
     * edit profile when an admin is editing
     *
     * @return int
     *
     */
    function datebook_get_current_user_id() {
        if( isset( $_GET['edit_user'] ) && !empty( $_GET['edit_user'] ) && current_user_can('edit_users') ){
			return (int)$_GET['edit_user'];
		} else {

			if (defined('DATEBOOK_CURRENT_USER_ID') && is_numeric(DATEBOOK_CURRENT_USER_ID)) {
				$current_user_id = DATEBOOK_CURRENT_USER_ID;
			} else {
				global $current_user;
				wp_get_current_user();
				$current_user_id = DateBook_Utilities::datebook_replace_user_by_admin($current_user->ID, 1);
			}

			return $current_user_id;
		}
    }


    /**
     * Get currency saved in the settings page
     *
     * @return string
     *
     */
    function datebook_get_active_currency() {

        $settings = get_option( 'datebook_settings' );

        return ( !empty( $settings['payments']['currency'] ) ? $settings['payments']['currency'] : 'USD' );

    }


    /*
     * Modify the logout url to redirect to current page if the user is in the front-end
     * and logs out
     *
     */
    function datebook_logout_redirect_url( $logout_url, $redirect ) {

        $current_page = datebook_get_current_page_url();

        // Do nothing if there's already a redirect in place
        if( !empty( $redirect ) )
            return $logout_url;

        // Do nothing if we're in an admin page
        if( strpos( $current_page, 'wp-admin' ) !== false )
            return $logout_url;

        $logout_url = add_query_arg( array( 'redirect_to' => urlencode( esc_url( $current_page ) ) ), $logout_url );

        return $logout_url;

    }
    add_filter( 'logout_url', 'datebook_logout_redirect_url', 10, 2 );


    /**
     * Add a notice on the DateBook Settings page if SSL is not being used, because PayPal IPN postback requires HTTPS enabled
     *
     */
    if  ( ! ( is_ssl() ) && isset( $_REQUEST['page'] ) && ( $_REQUEST['page'] == 'datebook-settings-page' ) )  {

        new DateBook_Add_General_Notices('datebook_ipn_requires_https',
                sprintf(esc_html__('PayPal uses Instant Payment Notification (IPN) which %1$s has been updated for increased security %2$s and now requires you to install an SSL certificate and enable HTTPS on your site in order to function properly. %3$sDismiss%4$s', 'datebook'), '<a href="https://www.paypal-knowledge.com/infocenter/index?page=content&widgetview=true&id=FAQ1916&viewlocale=en_US" target="_blank">', "</a>", "<a href='" . esc_url(add_query_arg('datebook_ipn_requires_https_dismiss_notification', '0')) . "'>", "</a>"),
                'update-nag');

    }


    /**
     * Adds a dismissable admin notice on all WordPress pages and a non-dismissable admin notice on DateBook's
     * Settings page requiring SSL to be enabled in order for all functionality to be available
     *
     */
    /*if( ! datebook_is_https() ) {

        $message = esc_html__( 'SSL is not enabled! You should install a valid SSL certificate.', 'datebook' );

        if( isset( $_REQUEST['page'] ) && $_REQUEST['page'] == 'datebook-settings-page' ) {

            new DateBook_Add_General_Notices( 'datebook_no_ssl_on_datebook_pages',
                $message,
                'update-nag');

        } else {

            new DateBook_Add_General_Notices( 'datebook_no_ssl',
                sprintf( $message . esc_html__( ' %1$sDismiss%2$s', 'datebook'), "<a href='" . esc_url(add_query_arg('datebook_no_ssl_stop_notification', '0')) . "'>", "</a>"),
                'update-nag');

        }

    }*/