<?php

/*-----------------------------------------------------------------------------------

	Plugin Name: Comments

-----------------------------------------------------------------------------------*/


// Add function to widgets_init that'll load our widget.
add_action( 'widgets_init', 'datebook_comments_widgets' );

// Comments widget.
function datebook_comments_widgets() {
	register_widget( 'DateBook_Comments_widget' );
}

// Widget class.
class DateBook_Comments_widget extends WP_Widget {


/*-----------------------------------------------------------------------------------*/
/*	Widget Setup
/*-----------------------------------------------------------------------------------*/
	
	function __construct() {
	
		/* Widget settings. */
		$widget_ops = array( 'classname' => 'datebook-comments-widget', 'description' => esc_html__( 'Displays comments that users left on profiles.', 'datebook' ) );

		parent::__construct( 'datebook_comments_widget', esc_html__( 'DateBook: Comments', 'datebook' ), $widget_ops );

	}


/*-----------------------------------------------------------------------------------*/
/*	Update Widget
/*-----------------------------------------------------------------------------------*/
	
	function update( $new_instance, $old_instance ) {
		$instance = $old_instance;

		/* Strip tags to remove HTML (important for text inputs). */
		$instance['title'] = strip_tags( $new_instance['title'] );
		$instance['comments_count'] = !empty($new_instance['comments_count']) ? $new_instance['comments_count'] : 5;
		$instance['comment_chars_amount'] = !empty($new_instance['comment_chars_amount']) ? $new_instance['comment_chars_amount'] : 120;
		$instance['sort_by'] = strip_tags($new_instance['sort_by']);
		$instance['enabled_image'] = !empty($new_instance['enabled_image']) ? 1 : 0;
		$instance['enabled_date'] = !empty($new_instance['enabled_date']) ? 1 : 0;

		return $instance;
	}
	

/*-----------------------------------------------------------------------------------*/
/*	Widget Settings
/*-----------------------------------------------------------------------------------*/
	 
	function form( $instance ) {

		/* Set up some default widget settings. */
		$defaults = array(
		'title' => 'Comments',
		);
		$instance = wp_parse_args( (array) $instance, $defaults );
		$comments_count = (isset($instance['comments_count']) && (int)$instance['comments_count'] > 0) ? (int)$instance['comments_count'] : 5;
		$comment_chars_amount = (isset($instance['comment_chars_amount']) && (int)$instance['comment_chars_amount'] > 0) ? (int)$instance['comment_chars_amount'] : 120;
		$sort_by = !empty($instance['sort_by']) ? strip_tags($instance['sort_by']) : 'descending';
		$enabled_image = isset($instance['enabled_image']) ? (bool) $instance['enabled_image'] : false;
		$enabled_date = isset($instance['enabled_date']) ? (bool) $instance['enabled_date'] : false;
		?>

		<p>
			<label for="<?php echo esc_attr($this->get_field_id( 'title' )); ?>"><?php esc_html_e('Title:', 'datebook' ) ?></label>
			<input type="text" class="widefat" id="<?php echo esc_attr($this->get_field_id( 'title' )); ?>" name="<?php echo esc_attr($this->get_field_name( 'title' )); ?>" value="<?php echo esc_attr($instance['title']); ?>" />
		</p>
		<p>
			<label for="<?php echo esc_attr($this->get_field_id('comments_count')); ?>"><?php esc_html_e('Number of Comments', 'datebook' ); ?></label>
			<input id="<?php echo esc_attr($this->get_field_id('comments_count')); ?>" name="<?php echo esc_attr($this->get_field_name('comments_count')); ?>" type="text" value="<?php echo esc_attr($comments_count); ?>" class="widefat" />
		</p>
		<p>
			<label for="<?php echo esc_attr($this->get_field_id('comment_chars_amount')); ?>"><?php esc_html_e('How many characters to show from each comment?', 'datebook' ); ?></label>
			<input id="<?php echo esc_attr($this->get_field_id('comment_chars_amount')); ?>" name="<?php echo esc_attr($this->get_field_name('comment_chars_amount')); ?>" type="text" value="<?php echo esc_attr($comment_chars_amount); ?>" class="widefat" />
		</p>
		<p>
			<label for="<?php echo esc_attr($this->get_field_id('sort_by')); ?>"><?php esc_html_e('Sort by date', 'datebook' );?></label>
			<?php $answers = array('DESC' => 'Descending','ASC' => 'Ascending','RAND' => 'Random'); ?>
			<select class="widefat" id="<?php echo esc_attr($this->get_field_id('sort_by')); ?>" name="<?php echo esc_attr($this->get_field_name('sort_by')); ?>">
				<?php foreach ($answers as $key => $answer ): ?>
				<option <?php if ($sort_by == $key ) { echo 'selected="selected"'; } ?> value="<?php echo sanitize_key($key); ?>"><?php echo esc_attr($answer); ?></option>
				<?php endforeach; ?>
			</select>
		</p>
		<p>
			<input type="checkbox"  id="<?php echo esc_attr($this->get_field_id('enabled_image'));?>" name="<?php echo esc_attr($this->get_field_name('enabled_image'));?>" <?php checked($enabled_image); ?> /> <?php esc_html_e("Show Image", 'datebook' );?>
		</p>
		<p>
			<input type="checkbox" id="<?php echo esc_attr($this->get_field_id('enabled_date'));?>" name="<?php echo esc_attr($this->get_field_name('enabled_date'));?>" <?php checked($enabled_date); ?> /> <?php esc_html_e("Show Date", 'datebook' );?>
		</p>
		<?php
	}


/*-----------------------------------------------------------------------------------*/
/*	Display Widget
/*-----------------------------------------------------------------------------------*/
	
	function widget( $args, $instance ) {
		extract( $args );

		/* Our variables from the widget settings. */
		$title = apply_filters('widget_title', $instance['title'] );
		$posts_per_page = ((int)$instance['comments_count'] > 0) ? (int)$instance['comments_count'] : 10;
		$comment_chars_amount = (isset($instance['comment_chars_amount']) && (int)$instance['comment_chars_amount'] > 0) ? (int)$instance['comment_chars_amount'] : 120;
		$enabled_image = isset($instance['enabled_image']) ? (bool)$instance['enabled_image']:0;
		$enabled_date = isset($instance['enabled_date']) ? (bool)$instance['enabled_date']:0;
		
		$sort_by = 'DESC';
		if(!empty($instance['sort_by']) && is_array($instance['sort_by'])):
			$sort_by =  array_filter($instance['sort_by']);
		elseif(!empty($instance['sort_by'])):
			$sort_by = explode(',',$instance['sort_by']);
		endif;
		
		/* Display Widget */
		if (of_get_option('enable_profile_comments') == 1) {

			/* Before widget (defined by themes). */
			echo $before_widget;

			?>

			<!-- sidebar -->
			<h4 class="mb-0 datebook-title-aside"><?php esc_html_e( 'Comments', 'datebook' ); ?></h4>

			<hr />

			<?php

			$args = array(
				'orderby' => 'comment_date_gmt',
				'order' => $sort_by[0],
				'status'  => 'approve',
				'meta_key' => 'comment_type',
			    'meta_value' => DATEBOOK_COMMENT_TYPE,
				'number'  => $posts_per_page,
				'no_found_rows' => false,
				'update_comment_meta_cache' => false
			);

			$comments = get_comments($args);


			if ( $comments ) :

				// Check if Retina enabled in Theme Options
				$enable_retina_images = of_get_option('enable_retina_images') ? 1 : 0;

				?>

				<ul class="list-unstyled">

				<?php

				foreach ( $comments as $comment ) {

					$profile_id = get_user_meta( $comment->user_id, 'profile_postid', true );
					$commenter_data = get_post($profile_id);
					$commenter_data_ID = isset($commenter_data->ID) ? $commenter_data->ID : 0;

					if($commenter_data_ID > 0){

						$gender = get_post_meta($commenter_data->ID, 'gender', true);
						$comment_post_time = DateBook_Utilities::datebook_date_i18n(2, $comment->comment_date);

						$thumbnail_url = DateBook_Utilities::datebook_profile_thumbnail($commenter_data_ID, 'image_comments_sidebar', $gender, true);

				?>
					<li class="media">

						<?php if ($enabled_image == 1) { ?>

						<a class="mt-1 mr-2" href="<?php echo esc_url(get_permalink($comment->comment_post_ID)); ?>?show=comments#commentid_<?php echo absint($comment->comment_ID); ?>"><img class="d-flex rounded" src="<?php echo esc_url( $thumbnail_url['img_full_url'] ); ?>" alt="" <?php if($enable_retina_images == 1) { ?>data-rjs="2"<?php } ?> /></a>

						<?php } ?>

						<div class="media-body">
						<h6 class="mt-0 mb-1"><a href="<?php echo esc_url(get_permalink($comment->comment_post_ID)); ?>?show=comments#commentid_<?php echo absint($comment->comment_ID); ?>"><?php echo esc_attr($commenter_data->post_title); ?></a></h6>
						<?php if ($enabled_date == 1) { ?>
						<p class="mb-1 small text-grey"><?php echo esc_attr($comment_post_time); ?></p>
						
						<?php } ?>

						<p class="mb-1">

						<?php 

						$excerpt = substr($comment->comment_content, 0, $comment_chars_amount);

						echo esc_attr($excerpt);

						if ((is_array($comment->comment_content) && count($comment->comment_content) > $comment_chars_amount) || $comment->comment_content > $comment_chars_amount) {

							echo '...';

						}

						?>

						</p>
						
						<?php 
						
						$my_comments_url = add_query_arg( array(
							'show' => 'comments'
						), get_permalink($comment->comment_post_ID) );

						echo ' <a class="btn btn-secondary btn-blue-grey btn-sm float-right" href="' . esc_url($my_comments_url) . '#commentid_' . absint($comment->comment_ID) . '">' . esc_html__( 'Read', 'datebook' ) . '</a>'; ?>

						</div>
						
						
					</li>

				<?php }} ?>

				</ul>

			<?php else: ?>

				<p><?php esc_html_e( 'No comments yet', 'datebook' ); ?></p>

			<?php

			endif;

			?>

			<!-- /sidebar -->

			<?php 

			/* After widget (defined by themes). */
			echo $after_widget;

		}

	}

}

?>