<?php
/**
 * Create Registration plans (Paid Subscriptions)
 *
 * @class DateBook_Register_Plans_Table
 * @author Pagick Themes
 * @since 1.6
 */
	 

/**
 * Define DateBook Table List
 * ============================================================================
 *
 * In this part you are going to define custom table list class,
 * that will display your database records in nice looking table
 *
 * http://codex.wordpress.org/Class_Reference/WP_List_Table
 * http://wordpress.org/extend/plugins/custom-list-table-example/
 */

if (!class_exists('WP_List_Table')) {

	require_once(ABSPATH . 'wp-admin/includes/class-wp-list-table.php');

}

/**
 * DateBook_Register_Plans_Table class that will display our custom table
 * records in nice table
 */
class DateBook_Register_Plans_Table extends WP_List_Table {

	/**
	 * [REQUIRED] You must declare constructor and give some basic params
	 */
	function __construct() {

		global $status, $page;

		parent::__construct(array(
			'singular' => 'field',
			'plural' => 'fields',
		));
	}


	/**
	 * [REQUIRED] This is the most important method
	 *
	 * It will get rows from database and prepare them to be showed in table
	 */
	function get_reserved_plan() {

		$get_reserved_plan = array('00000000000001' => array(
			'subscription_type' => 'feature',
			'slug' => '00000000000001',
			'subscription_id' => '00000000000001',
			'active' => '1',
			'name_en_US' => 'All features',
			'plan_description_en_US' => 'Features used as standalone apart from any subscription plan',
			'activate_by_default' => 0,
			'hide_after_expire' => 0,
			'wc_subscription_product' => '',
			'price' => array('0' => 0),
			'register_period_number' => array('0' => 1),
			'register_period' => array('0' => 2),
			'pointsquantity' => 5,
			'recurring' => 0,
			'free_for_gender' => array('0' => 0),
			'top_position' => 1,
			'times_top_position' => 1,
			'period_number_top_position' => 1,
			'period_top_position' => 3,
			'period_long_number_top_position' => 1,
			'period_long_top_position' => 3,
			'firstactivate_top_position' => 0,
			'suggest_top_position' => 1,
			'points_for_top_position' => 5,
			'featured_position' => 1,
			'times_featured_position' => 1,
			'period_number_featured_position' => 1,
			'period_featured_position' => 3,
			'period_long_number_featured_position' => 1,
			'period_long_featured_position' => 3,
			'firstactivate_featured_position' => 0,
			'suggest_featured_position' => 1,
			'points_for_featured_position' => 5,
			'simultaneous_chatters' => 1,
			'quantity_simultaneous_chatters' => 10,
			'period_number_simultaneous_chatters' => 1,
			'period_simultaneous_chatters' => 3,
			'points_for_one_message' => 1,
			'suggest_simultaneous_chatters' => 1,
			'enter_phone' => 1,
			'points_for_enter_phone' => 5,
			'enter_website' => 1,
			'points_for_enter_website' => 5,
			'subscription_limitphotos' => 20,
			'points_for_one_image' => 1
			)
        );

		return $get_reserved_plan;

	}


	/**
	 * [REQUIRED] this is a default column renderer
	 *
	 * @param $item - row (key, value array)
	 * @param $column_name - string (key)
	 * @return HTML
	 */
	function column_default($item, $column_name) {
		
		if ($column_name == 'subscription_type'){

			// if foreign language do not exist, then show title in English
			$item[$column_name] = isset($item[$column_name]) && $item[$column_name] == 'feature' ? '<span style="background-color:#5F9EA0; color:white; padding: 3px 5px 4px; border-radius: 5px">' . esc_html__('Features', 'datebook') . '</span>' : '<span style="background-color:#DEB887; color:white; padding: 3px 5px 4px; border-radius: 5px">' . esc_html__('Subscription', 'datebook') . '</span>';

			return $item[$column_name];

		} else {

			return esc_attr($item[$column_name]);

		}

	}


	/**
	 * [OPTIONAL] this is example, how to render specific column
	 *
	 * method name must be like this: "column_[column_name]"
	 *
	 * @param $item - row (key, value array)
	 * @return HTML
	 */
	function column_active($item) {
		
		if (isset($item['active']) && $item['active'] == 1) {
			
			$active_out = '<span style="color:green">' . esc_html__('Yes', 'datebook') . '</span>';
			
		} else {
			
			$active_out = '<span style="color:red">' . esc_html__('No', 'datebook') . '</span>';
			
		}

		return $active_out;

	}


	/**
	 * [OPTIONAL] this is example, how to render column with actions,
	 * when you hover row "Edit | Delete" links showed
	 *
	 * @param $item - row (key, value array)
	 * @return HTML
	 */
	function column_name($item) {
		
		$menu = Options_Framework_Admin::menu_settings();

		if ( isset( $menu['menu_register_plans_slug'] ) ) {
			$menu_register_plans_slug = $menu['menu_register_plans_slug'];
		}

		// links going to /admin.php?page=[your_plugin_page][&other_params]
		// notice how we used $_REQUEST['page'], so action will be done on current page
		// also notice how we use $this->_args['singular'] so in this example it will
		// be something like &register_plan=2
		$actions = array(
			'edit' => sprintf('<a href="admin.php?page=%s&subpage=edit_form&subscription_id=%s&slug=%s">%s</a>', esc_attr($menu_register_plans_slug), esc_attr($item['subscription_id']), esc_attr($item['slug']), esc_html__('Edit', 'datebook')),
			'delete' => sprintf('<a onclick="return confirm(\' %s \')" href="admin.php?page=%s&action=delete&subscription_id=%s&slug=%s">%s</a>', esc_html__('Do you really want to delete it?', 'datebook'), esc_attr($menu_register_plans_slug), esc_attr($item['subscription_id']), esc_attr($item['slug']), esc_html__('Delete', 'datebook')),
		);

		// if foreign language do not exist, then show title in English
		$item['name'] = isset($item['name_' . DATEBOOK_WPLANG]) && !empty($item['name_' . DATEBOOK_WPLANG]) ? $item['name_' . DATEBOOK_WPLANG] : (isset($item['name_en_US']) ? $item['name_en_US'] : '');

		return sprintf('%s %s',
			esc_attr($item['name']),
			$this->row_actions($actions)
		);
	}


	/**
	 * [REQUIRED] This method return columns to display in table
	 * you can skip columns that you do not want to show
	 * like content, or description
	 *
	 * @return array
	 */
	function get_columns() {

		$columns = array(
			'name' => esc_html__('Name', 'datebook'),
			'subscription_type' => esc_html__('Subscription type', 'datebook'),
			'slug' => esc_html__('ID', 'datebook'),
			'active' => esc_html__('Enable', 'datebook')
		);

		return $columns;

	}


	/**
	 * [OPTIONAL] This method return columns that may be used to sort table
	 * all strings in array - is column names
	 * notice that true on name column means that its default sort
	 *
	 * @return array
	 */
	function get_sortable_columns() {

		$sortable_columns = array(
			'name' => array('name', true),		
		);

		return $sortable_columns;

	}


	/**
	 * [OPTIONAL] This method processes bulk actions
	 * it can be outside of class
	 * it can not use wp_redirect coz there is output already
	 * in this example we are processing delete action
	 * message about successful deletion will be shown on page in next part
	 */
	function process_bulk_action() {

		if(isset($_REQUEST['action']) && $_REQUEST['action'] == 'delete'){

			$get_slug = isset($_REQUEST['subscription_id']) ? $_REQUEST['subscription_id'] : (isset($_REQUEST['slug']) ? $_REQUEST['slug'] : '');

			if (isset($get_slug)) {

				// Update entire array
				// Get entire array
				$get_subscription_plans = get_option('datebook_subscription_plans');

				if ('delete' === $this->current_action()) {

					foreach ($get_subscription_plans as $key => $value) {

						// Remove field from array
						if ((isset($_REQUEST['subscription_id']) && $_REQUEST['subscription_id'] == $key) || (isset($_REQUEST['slug']) && $_REQUEST['slug'] == $key)) {
							unset($get_subscription_plans[$key]);
						}

					}

					//Update entire array
					update_option('datebook_subscription_plans', $get_subscription_plans);

				}
			}
		}
	}


	/**
	 * It will get rows from database and prepare them to be showed as select
	 */
	function get_all_register_plans($selected_value, $select_name) {
		
		global $period_list_array;

		$layout = '';
		$layout .= '<select name="' . esc_attr($select_name) . '">';
		
		foreach($period_list_array as $key => $value){

			if ($selected_value == $key) { $selected = ' selected="selected" '; } else { $selected = ''; }

			$layout .= '<option value="' . esc_attr($key) . '" '.$selected.'>' . esc_attr($value) . '</option>';
		}
		$layout .= '</select>';
		
		return $layout;

    }


	/**
	 * It will get recurring methods and prepare them to be showed as select
	 */
	function get_recurring_select($selected_value, $select_name) {
		
		$list_array = array(
			'0' => esc_html__('No recurring', 'datebook'), 
			'1' => esc_html__('Offer the recurring but do not force it by default', 'datebook'),
			'2' => esc_html__('Force the recurring by default', 'datebook'), 
			
		);
		
		$layout = '';
		$layout .= '<select name="' . $select_name . '">';
		
		foreach($list_array as $key => $value){

			if ($selected_value == $key) { $selected = ' selected="selected" '; } else { $selected = ''; }

			$layout .= '<option value="' . esc_attr($key) . '" '.$selected.'>' . esc_attr($value) . '</option>';
		}
		$layout .= '</select>';
		
		return $layout;

    }


	/**
	 * Get all WooCommerce subscription plans
	 */
	function get_all_wc_subscription_plans($register_plan_register_period, $register_select_name) {
		
		$register_plans_out = '';

		$args = array(
			'post_type' => 'product',
			'post_status' => 'publish',
			'posts_per_page' => 100,
			'meta_key' => '_datebook_subscribe',
			'meta_value' => 'yes',
			'meta_compare' => '='
		);

		$wc_query = new WP_Query($args);

		if ($wc_query->have_posts()) :

		$register_plans_out .= '<select name="' . $register_select_name . '">';
		
		while ($wc_query->have_posts()) : $wc_query->the_post();

			$get_the_ID = get_the_ID();
			$get_the_title = get_the_title();

			if ($register_plan_register_period == $get_the_ID) { $selected = ' selected="selected" '; } else { $selected = ''; }

			$register_plans_out .= '<option value="' . esc_attr($get_the_ID) . '" '.$selected.'>' . esc_attr($get_the_title) . '</option>';
		
		endwhile;
	
		wp_reset_postdata();

		$register_plans_out .= '</select>';

		else:

			$register_plans_out .= esc_html__( 'No WooCommerce product found here: WordPress -> Products', 'datebook' );

		endif;


		return $register_plans_out;

    }


	/**
	 * [REQUIRED] This is the most important method
	 *
	 * It will get rows from database and prepare them to be showed in table
	 */
	function prepare_items() {

		$per_page = 300; // constant, how much records will be shown per page

		$columns = $this->get_columns();
		$hidden = array();
		$sortable = $this->get_sortable_columns();

		// here we configure table headers, defined in our methods
		$this->_column_headers = array($columns, $hidden, $sortable);

		// [OPTIONAL] process bulk action if any
		$this->process_bulk_action();

		$get_subscription_plans = get_option('datebook_subscription_plans') ? get_option('datebook_subscription_plans') : array();
		
		$get_reserved_plans = $this->get_reserved_plan();
		
		$all_plans = array_merge($get_subscription_plans,$get_reserved_plans);

		if(is_array($all_plans) && !empty($all_plans)){
		
			// will be used in pagination settings
			$total_items = count($all_plans);

			// prepare query params, as usual current page, order by and order direction
			$paged = isset($_REQUEST['paged']) ? ($per_page * max(0, intval($_REQUEST['paged']) - 1)) : 0;

			// Define $items array
			// notice that last argument is ARRAY_A, so we will retrieve array
			$this->items = $all_plans;

			// configure pagination
			$this->set_pagination_args(array(
				'total_items' => $total_items, // total items defined above
				'per_page' => $per_page, // per page constant defined at top of method
				'total_pages' => ceil($total_items / $per_page) // calculate pages count
			));
		}
	}
}


class Options_Framework_Subscription_Plans {

	/**
	 * List page handler
	 *
	 * This function renders our custom table
	 * Notice how we display message about successfull deletion
	 * Actualy this is very easy, and you can add as many features
	 * as you want.
	 *
	 * Look into /wp-admin/includes/class-wp-*-list-table.php for examples
	 */
	static function datebook_page_handler() {

		if (isset($_REQUEST['subpage']) && $_REQUEST['subpage'] == 'edit_form') {

			self::display_register_plans_page();

		} else {

			$table = new DateBook_Register_Plans_Table();
			$table->prepare_items();

			$message = '';
			if ('delete' === $table->current_action()) {
				$message = esc_html__('Item deleted', 'datebook');
			}

			$menu = Options_Framework_Admin::menu_settings();

			if ( isset( $menu['menu_register_plans_slug'] ) ) {
				$menu_register_plans_slug = $menu['menu_register_plans_slug'];
			}

			?>

			<div class="wrap">

				<h1 class="wp-heading-inline"><?php esc_html_e('Subscription plans', 'datebook'); ?></h1>
				
					<a class="page-title-action" href="<?php echo get_admin_url(get_current_blog_id(), 'admin.php?page=' . $menu_register_plans_slug . '&subpage=edit_form'); ?>"><?php esc_html_e('Add plan', 'datebook'); ?></a>
			
					<hr class="wp-header-end">
			
					<div id="message" class="bg-white">
						<p><?php esc_html_e('On this page you can add, edit, delete Subscription plans.', 'datebook'); ?></p>
					</div>
			
					<?php 
				
					if (!empty($message)) {

						echo '<div id="message" class="updated below-h2"><p> ' . esc_attr($message) . '</p></div>';

					}

					?>

					<form id="register_plans-table" method="POST">
					
						<input type="hidden" name="page" value="<?php if ( isset($menu_register_plans_slug) ) { echo esc_attr($menu_register_plans_slug); } ?>"/>

						<?php $table->display(); ?>

					</form>

				</div>

				<?php
			
		}
	}


	/**
	 * Form for adding andor editing row
	 * ============================================================================
	 *
	 * In this part you are going to add admin page for adding andor editing items
	 * You cant put all form into this function, but in this example form will
	 * be placed into meta box, and if you want you can split your form into
	 * as many meta boxes as you want
	 *
	 * http://codex.wordpress.org/Data_Validation
	 * http://codex.wordpress.org/Function_Reference/selected
	 */

	/**
	 * Form page handler checks is there some data posted and tries to save it
	 * Also it renders basic wrapper in which we are callin meta box render
	 */
	static function display_register_plans_page() {

		$message = '';
		$notice = '';

		// here we are verifying does this request is post back and have correct nonce
		if (isset($_POST['nonce']) && wp_verify_nonce($_POST['nonce'], basename(__FILE__))) {
			
			// validate data, and if all ok save item to database
			// if id is zero insert otherwise update
			$item_data = self::validate_data($_POST);

			if ($item_data['success'] === true) {

				$item = $item_data['form_fields'];

				//Get entire array
				$get_subscription_plans = get_option('datebook_subscription_plans');

				// Get reserved features
				$reserved_plan_array = new DateBook_Register_Plans_Table();
				$get_reserved_plan = $reserved_plan_array->get_reserved_plan();

				if (isset($item['subscription_id']) && isset($get_subscription_plans[$item['subscription_id']]['subscription_id']) && !empty($get_subscription_plans[$item['subscription_id']]['subscription_id']) && $get_subscription_plans[$item['subscription_id']]['subscription_id'] > 0) {
					$item['subscription_id'] = $get_subscription_plans[$item['subscription_id']]['subscription_id'];
				} else if (isset($item['slug']) && isset($get_subscription_plans[$item['slug']]['subscription_id']) && !empty($get_subscription_plans[$item['slug']]['subscription_id']) && $get_subscription_plans[$item['slug']]['subscription_id'] > 0) {
					$item['subscription_id'] = $get_subscription_plans[$item['slug']]['subscription_id'];
				} else if (isset($item['slug']) && isset($get_reserved_plan[$item['slug']]['subscription_id']) && !empty($get_reserved_plan[$item['slug']]['subscription_id']) && $get_reserved_plan[$item['slug']]['subscription_id'] > 0) {
					$item['subscription_id'] = $get_reserved_plan[$item['slug']]['subscription_id'];
				} else {
					$item['subscription_id'] = rand(1000, 9999) . time();
				}


				$item['slug'] = $item['subscription_id'];

				//Alter the options array appropriately
				$get_subscription_plans[$item['subscription_id']] = $item;

				//Update entire array
				$result = update_option('datebook_subscription_plans', $get_subscription_plans);

				if( $result === FALSE){
					$notice = esc_html__('Oops! Something went wrong. Data has NOT been saved.', 'datebook');
				} else {
					$message = esc_html__('Plan has been successfully saved', 'datebook');
				}


			} else {

				// if $item_data not true it contains error message(s)
				$notice = $item_data['error_message'];

			}

		} else {

			// if this is not post back we load item to edit or give new one to create
			$item = array();
			
			if (isset($_REQUEST['subscription_id']) || isset($_REQUEST['slug'])) {

				//Get entire array
				$get_subscription_plans = get_option('datebook_subscription_plans');
				
				//Alter the options array appropriately
				if(isset($get_subscription_plans[$_REQUEST['slug']])){
					$item = $get_subscription_plans[$_REQUEST['slug']];
					if(isset($item['subscription_type']) && $item['subscription_type'] == 'feature'){
						$item['subscription_type'] = 'feature';
					} else {
						$item['subscription_type'] = 'subscription';
					}
				} else if(isset($get_subscription_plans[$_REQUEST['subscription_id']])){
					$item = $get_subscription_plans[$_REQUEST['subscription_id']];
					if(isset($item['subscription_type']) && $item['subscription_type'] == 'feature'){
						$item['subscription_type'] = 'feature';
					} else {
						$item['subscription_type'] = 'subscription';
					}
				}


				if (!$item) {

					// Get reserved features
					$reserved_plan_array = new DateBook_Register_Plans_Table();
					$get_reserved_plan = $reserved_plan_array->get_reserved_plan();

					if(isset($get_reserved_plan[$_REQUEST['slug']])){
						$item = $get_reserved_plan[$_REQUEST['slug']];
						$item['subscription_type'] = 'feature';
					}

					if (!$item) {

						$item = isset($item_data['form_fields']) ? $item_data['form_fields'] : '';
						$notice = esc_html__('Item not found', 'datebook');

					}
				}
			} else {
				$item['subscription_type'] = 'subscription';
			}
		}

		// here we adding our custom meta box
		add_meta_box('register_plans_form_meta_box', 'Edit', array( __CLASS__ , 'datebook_register_plans_edit'), 'register_plan', 'normal', 'default');

		$menu = Options_Framework_Admin::menu_settings();

		$menu_register_plans_slug = ( isset( $menu['menu_register_plans_slug'] ) ) ? $menu['menu_register_plans_slug'] : '';

    ?>

		<div class="wrap">

			<h1 class="wp-heading-inline"><?php echo esc_html__('Edit', 'datebook')?></h1>

			<a class="page-title-action" href="<?php echo get_admin_url(get_current_blog_id(), 'admin.php?page='.esc_attr($menu_register_plans_slug));?>"><?php esc_html_e('Back to Subscription plans', 'datebook')?></a> 
			
			<a class="page-title-action" href="<?php echo get_admin_url(get_current_blog_id(), 'admin.php?page=' . $menu_register_plans_slug . '&subpage=edit_form');?>"><?php esc_html_e('Add plan', 'datebook')?></a>

			<hr class="wp-header-end">

			<?php if (!empty($notice)): ?>

				<div id="notice" class="error">
					<p><?php echo esc_attr($notice); ?></p>
				</div>

			<?php endif;?>

			<?php if (!empty($message)): ?>

				<div id="message" class="updated">
					<p><?php echo esc_attr($message) ?></p>
				</div>

			<?php endif;?>

			<form id="form" method="POST">

				<input type="hidden" name="nonce" value="<?php echo wp_create_nonce(basename(__FILE__))?>"/>
				
				<?php if (isset($item['slug']) && !empty($item['slug'])) { ?>

					<input type="hidden" name="slug" value="<?php echo esc_attr($item['slug']); ?>"/>

				<?php } ?>

				<?php if (isset($item['subscription_id']) && !empty($item['subscription_id'])) { ?>

					<input type="hidden" name="subscription_id" value="<?php echo esc_attr($item['subscription_id']); ?>"/>

				<?php } ?>

				<?php if (isset($item['subscription_type']) && !empty($item['subscription_type'])) { ?>

					<input type="hidden" name="subscription_type" value="<?php echo esc_attr($item['subscription_type']); ?>"/>
				
				<?php } ?>

				<div class="metabox-holder" id="poststuff">
					<div id="post-body">
						<div id="post-body-content">

							<?php /* And here we call our custom meta box */ ?>
							<?php do_meta_boxes('register_plan', 'normal', $item); ?>

							<input id="submit" class="button-primary" type="submit" name="submit" value="<?php esc_html_e('Save', 'datebook')?>" />

						</div>
					</div>
				</div>

			</form>

		</div>

	<?php

	}

	/**
	 * This function renders our custom meta box
	 * $item is row
	 *
	 * @param $item
	 */
	static function datebook_register_plans_edit($item) {

		global $wp_locale_all, $paid_features_subscriptions;

		// get languages
		$languages_array = array();

		$languages_array['en_US'] = esc_html__( 'English', 'datebook' );

		$languages = DateBook_Localization::datebook_all_available_language_files();

		foreach($languages as $language) {
			if($language['checked'] == 1){
				$languages_array[$language['locale']] = $wp_locale_all[$language['locale']];
			}
		}


		$features_array = array();
		if ($paid_features_subscriptions) {
			foreach ($paid_features_subscriptions as $key => $section) {
				$features_array[$key] = esc_html__( $section, 'datebook' );
			}
		}

		?>

		<table cellspacing="2" cellpadding="5" style="width: 100%;" class="form-table">
			<tbody>

				<?php if ($item['subscription_type'] == 'subscription'){ ?>

				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="active"><?php esc_html_e('Enable', 'datebook'); ?></label>
					</th>
					<td><input id="datebook-active"  name="active" value="1" type="checkbox" <?php if (isset($item['active']) && $item['active'] == 1) { echo ' checked="checked"'; } ?> /><label for="datebook-active"><?php esc_html_e('Yes', 'datebook'); ?></label></td>
				</tr>

				<?php foreach ( $languages_array as $language_key => $language_name ) {
					
					if($language_key == 'en_US') {
						$required = 'required';
					} else {
						$required = '';
					}
						
					
					?>
					
					<tr class="form-field">
					<th valign="top" scope="row">
						<label for="name"><?php echo sprintf( esc_html__( 'Title (%1$s)', 'datebook' ), $language_name ) ?></label>
					</th>
					<td><input id="name" name="name_<?php echo esc_attr($language_key); ?>" type="text" style="width: 100%" value="<?php if (isset($item['name_' . $language_key])) { echo esc_attr($item['name_' . $language_key]); } ?>" size="50" class="code" placeholder="<?php echo sprintf( esc_html__( 'Enter title in %1$s', 'datebook' ), esc_attr($language_name) ) ?>" <?php echo esc_attr($required); ?> /></td>
				</tr>
					
					<tr class="form-field">
					<th valign="top" scope="row">
						<label for="datebook-textarea"><?php echo sprintf( esc_html__( 'Short description (%1$s)', 'datebook' ), esc_attr($language_name) ) ?></label>
					</th>
					<td>

						<textarea id="datebook-textarea" name="plan_description_<?php echo esc_attr($language_key); ?>" rows="2" style="width:100%"><?php if (!empty($item['plan_description_' . $language_key])) { echo esc_attr($item['plan_description_' . $language_key]); } ?></textarea>
						
					</td>
				</tr>
					
				<?php } ?>


				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="datebook-activate-by-default"><?php esc_html_e('Subscribe to this plan by default', 'datebook'); ?></label>
					</th>
					<td>
					<input id="datebook-activate-by-default"  name="activate_by_default" value="1" type="checkbox" <?php if (isset($item['activate_by_default']) && $item['activate_by_default'] == 1) { echo ' checked="checked"'; } ?> /><label for="datebook-activate-by-default"><?php esc_html_e('Yes', 'datebook'); ?>. <?php esc_html_e('Member will be subscribed to this plan automatically after registration on the website', 'datebook'); ?>.</label><br>
					<input id="datebook-hide-by-default"  name="hide_by_default" value="1" type="checkbox" <?php if (isset($item['hide_by_default']) && $item['hide_by_default'] == 1) { echo ' checked="checked"'; } ?> /><label for="datebook-hide-by-default"><?php esc_html_e('Hide', 'datebook'); ?>. <?php esc_html_e('Do not show this plan to a member if he subscribed to it by default', 'datebook'); ?>.</label>
					</td>
				</tr>

				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="datebook-hide-after-expire"><?php esc_html_e('Hide this plan after expiration', 'datebook'); ?></label>
					</th>
					<td><input id="datebook-hide-after-expire"  name="hide_after_expire" value="1" type="checkbox" <?php if (isset($item['hide_after_expire']) && $item['hide_after_expire'] == 1) { echo ' checked="checked"'; } ?> /><label for="datebook-hide-after-expire"><?php esc_html_e('Yes', 'datebook'); ?>. <?php esc_html_e('If a member has subscribed to this plan, he will not be able to subscribe to this plan a second time after it expires', 'datebook'); ?>.</label></td>
				</tr>

				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="wc_subscription_product"><?php esc_html_e('WooCommerce Subscription', 'datebook'); ?></label>
					</th>
					<td>
					
					<?php
					
					if ( class_exists( 'WooCommerce' ) ) {
						
						$item_wc_subscription_product = isset($item['wc_subscription_product']) ? $item['wc_subscription_product'] : '';
						
						$all_wc_subscription_plans = new DateBook_Register_Plans_Table;
						echo $all_wc_subscription_plans->get_all_wc_subscription_plans($item_wc_subscription_product, 'wc_subscription_product');

					?>
					
                    <?php echo esc_html__('Select a WooCommerce product. It will be connected to this plan', 'datebook'); ?>
                    
					<?php } else { ?>
                    
                    	<?php esc_html_e('WooCommerce is not installed and activated', 'datebook'); ?>
                    
                    <?php } ?>

					</td>
				</tr>


				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="datebook-price"><?php echo esc_html__( 'Price', 'datebook' ); ?> + <?php esc_html_e('Period', 'datebook'); ?></label>
						<br />
						<small><?php esc_html_e('What currency is used you can look at the Dashboard -> Payment tab', 'datebook'); ?></small>
					</th>

					<td>
					<table id="datebook-pricetable" cellspacing="0" cellpadding="0" class="form-table" style="width:100%" bgcolor="#EDEADE">

					<?php 

					if(isset($item['price']) && is_array($item['price'])) {

						foreach ( $item['price'] as $price_key => $price_value ) {
						
						?>

						<tr>
							<td>

							<label for="datebook-price"><?php echo esc_html__( 'Price', 'datebook' ); ?>:</label> &nbsp; 
							<input id="datebook-price" name="price[]" type="text" style="width: 120px" value="<?php if (isset($price_value)) { echo esc_attr($price_value); } ?>" size="5" maxlength="10" class="code" placeholder="<?php echo esc_html__( 'Enter number', 'datebook' ); ?>" required="required" /> <?php esc_html_e('Enter ONLY number', 'datebook'); ?>
					
							<a class="datebook-removeprice alignright" href="javascript:void(0)"><?php esc_html_e('remove', 'datebook'); ?></a>
							
							<div class="clear"></div>
					
							<label for="datebook-period"><?php echo esc_html__( 'Period', 'datebook' ); ?>:</label> 
							<input id="register_period_number" name="register_period_number[]" type="text" style="width: 120px" value="<?php if (isset($item['register_period_number'][$price_key])) { echo esc_attr($item['register_period_number'][$price_key]); } ?>" size="3" class="code" placeholder="<?php esc_html_e('Enter number', 'datebook'); ?>" required="required" />
						
							<?php
							
							if (isset($item['register_period'][$price_key])) {
						
								$all_register_plans_register_periods = new DateBook_Register_Plans_Table;
								echo $all_register_plans_register_periods->get_all_register_plans($item['register_period'][$price_key], 'register_period[]');
							
							}

							?>

							</td>
						</tr>
					
						<?php } ?>

					<?php } else { ?>
					
					<tr>
							<td>

							<label for="datebook-price"><?php echo esc_html__( 'Price', 'datebook' ); ?>:</label> &nbsp; 
							<input id="datebook-price" name="price[]" type="text" style="width: 120px" value="<?php if (isset($price_value)) { echo esc_attr($price_value); } ?>" size="5" maxlength="5" class="code" placeholder="<?php echo esc_html__( 'Enter number', 'datebook' ); ?>" /> <?php esc_html_e('Enter ONLY number.', 'datebook'); ?>
					
							<a class="datebook-removeprice" href="javascript:void(0)"><?php esc_html_e('Remove', 'datebook'); ?></a>
							
							<div class="clear"></div>
					
							<label for="datebook-period"><?php echo esc_html__( 'Period', 'datebook' ); ?>:</label> 
							<input id="register_period_number" name="register_period_number[]" type="text" style="width: 120px" value="<?php if (isset($item['register_period_number'])) { echo esc_attr($item['register_period_number']); } ?>" size="3" class="code" placeholder="<?php esc_html_e('Enter number', 'datebook'); ?>" />
						
							<?php
					
							if (isset($item['register_period'])) {
								$item_register_period = esc_attr($item['register_period']);
							} else {
								$item_register_period = '';
							}

							$all_register_plans_register_periods = new DateBook_Register_Plans_Table;
							echo $all_register_plans_register_periods->get_all_register_plans($item_register_period, 'register_period[]');

							?>

							</td>
						</tr>
					
					<?php } ?>
					
					</table>
					
					<a id="datebook-addprice" href="javascript:void(0)"><?php esc_html_e('Add more', 'datebook'); ?></a> 
					<span class="alignright"><?php esc_html_e('0 = unlimited', 'datebook'); ?></span>
					
					</td>
					
				</tr>

				<tr class="form-field" style="<?php if (of_get_option('points_system') == 0) { echo 'display:none'; } ?>">
					<th valign="top" scope="row">
						<label for="datebook-pointsquantity"><?php echo esc_html__( 'Points', 'datebook' ); ?></label>
					</th>
					<td><input id="datebook-pointsquantity" name="pointsquantity" type="text" style="width: 150px" value="<?php if (isset($item['pointsquantity'])) { echo esc_attr($item['pointsquantity']); } else { echo '50'; } ?>" size="5" maxlength="5" class="code" placeholder="<?php echo esc_html__( 'Enter number', 'datebook' ); ?>" /> <?php esc_html_e('How many points will be automatically added after paying for this plan?', 'datebook'); ?>

					</td>
				</tr>
				
				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="active"><?php esc_html_e('Recurring payment', 'datebook'); ?></label>
					</th>
					<td>
                    
					<?php
					
					if (isset($item['recurring'])) {
						$item_recurring = esc_attr($item['recurring']);
					} else {
						$item_recurring = 0;
					}

					$all_recurring_select = new DateBook_Register_Plans_Table;
					echo $all_recurring_select->get_recurring_select($item_recurring, 'recurring');

					?>
                    
                    </td>
				</tr>
				
				
				<tr class="form-field">
					<th valign="top" scope="row">
						<label for="active"><?php esc_html_e('Free for genders', 'datebook'); ?></label>
					</th>
					<td>
					
					<p><?php esc_html_e('Select genders who must not pay for this subscription plan. They will be able to activate this plan for free.', 'datebook'); ?></p>
                    
					<?php
					
					global $datebook_gender_titles;
					
					// get genders
					$sex_titles_array = array();

					foreach ( $datebook_gender_titles as $key => $value ) {

						//$sex_titles_array[$key] = esc_html__( $value, 'datebook' );

						?>

						<input id="datebook-free-for-gender-<?php echo esc_attr($key); ?>" name="free_for_gender[<?php echo esc_attr($key); ?>]" type="checkbox" value="1" <?php if (isset($item['free_for_gender'][$key]) && $item['free_for_gender'][$key] == 1) { echo ' checked="checked"'; } ?> /> <label for="datebook-free-for-gender-<?php echo esc_attr($key); ?>"><?php esc_html_e($value, 'datebook'); ?></label><br>
						
					<?php } ?>

                    </td>
				</tr>
				
				<?php } ?>
				
				

				<tr class="form-field">
					<th><label for="datebook-features"><?php echo esc_html__( 'Features', 'datebook' ); ?></label></th>
					<td>

						<table cellspacing="0" cellpadding="0" class="form-table" style="width:100%" bgcolor="#EDEADE">

							<?php if ($item['subscription_type'] == 'subscription'){ ?>
							<tr class="form-field">
								<td><?php esc_html_e('this plan incudes', 'datebook'); ?>:</td>
							</tr>
							<?php } ?>
							
							
							<?php foreach ( $features_array as $feature_key => $feature_name ) { ?>

							<tr class="form-field">
								<td>

								<input id="datebook-<?php echo esc_attr($feature_key); ?>" name="<?php echo esc_attr($feature_key); ?>" type="checkbox" value="1" <?php if (isset($item[$feature_key]) && $item[$feature_key] == 1) { echo ' checked="checked"'; } ?> /> <label for="datebook-<?php echo esc_attr($feature_key); ?>"><?php esc_html_e($feature_name, 'datebook'); ?></label>


								<?php if (isset($feature_key) && ($feature_key == 'top_position' || $feature_key == 'featured_position')) { ?>

								<table cellspacing="0" cellpadding="0" class="form-table" bgcolor="#F7F5EE">

								<tr>
									<td>

									<?php esc_html_e('User can place his profile on ', 'datebook'); ?> <?php esc_html_e($feature_name, 'datebook'); ?>

									<div class="clear"></div>									

									<label for="datebook-times_<?php echo esc_attr($feature_key); ?>"><?php echo esc_html__( 'Up to', 'datebook' ); ?>:</label> &nbsp; 
									<input name="times_<?php echo esc_attr($feature_key); ?>" type="text" style="width: 120px" value="<?php if (isset($item['times_' . $feature_key]) && $item['times_' . $feature_key] > 0) { echo esc_attr($item['times_' . $feature_key]); } ?>" size="5" maxlength="5" placeholder="<?php echo esc_html__( 'Enter number', 'datebook' ); ?>" /> <?php esc_html_e('times', 'datebook'); ?>

									<div class="clear"></div>

									<label for="datebook-period_<?php echo esc_attr($feature_key); ?>"><?php echo esc_html__( 'Not often than 1 time every', 'datebook' ); ?>:</label> 
									<input name="period_number_<?php echo esc_attr($feature_key); ?>" type="text" style="width: 120px" value="<?php if (isset($item['period_number_' . $feature_key])) { echo esc_attr($item['period_number_' . $feature_key]); } ?>" size="3" placeholder="<?php esc_html_e('Enter number', 'datebook'); ?>" />
						
									<?php

									if (isset($item['period_' . $feature_key])) {
										$item_period_feature_key = $item['period_' . $feature_key];
									} else {
										$item_period_feature_key = '';
									}

									$all_periods_position = new DateBook_Register_Plans_Table;
									echo $all_periods_position->get_all_register_plans($item_period_feature_key, 'period_' . $feature_key);

									?>
									
									<div class="clear"></div>

									<label for="datebook-period-long-<?php echo esc_attr($feature_key); ?>"><?php echo esc_html__( 'Period lasts', 'datebook' ); ?>:</label> 
									<input name="period_long_number_<?php echo esc_attr($feature_key); ?>" type="text" style="width: 120px" value="<?php if (isset($item['period_long_number_' . $feature_key])) { echo esc_attr($item['period_long_number_' . $feature_key]); } ?>" size="3" placeholder="<?php esc_html_e('Enter number', 'datebook'); ?>" />
						
									<?php

									if (isset($item['period_long_' . $feature_key])) {
										$item_period_long_feature_key = $item['period_long_' . $feature_key];
									} else {
										$item_period_long_feature_key = '';
									}

									echo $all_periods_position->get_all_register_plans($item_period_long_feature_key, 'period_long_' . $feature_key);

									?>

									</td>
								</tr>


								<tr>
									<td>

									<input id="datebook-firstactivate-<?php echo esc_attr($feature_key); ?>" name="firstactivate_<?php echo esc_attr($feature_key); ?>" type="checkbox" value="1" <?php if (isset($item['firstactivate_' . $feature_key]) && $item['firstactivate_' . $feature_key] == 1) { echo ' checked="checked"'; } ?> /> <label for="datebook-firstactivate-<?php echo esc_attr($feature_key); ?>"><?php esc_html_e('Activate as soon as this plan is selected. Otherwise, first time a user has to activate it manually.', 'datebook'); ?></label>

									<div class="clear"></div>

									<input id="datebook-suggest-<?php echo esc_attr($feature_key); ?>" name="suggest_<?php echo esc_attr($feature_key); ?>" type="checkbox" value="1" <?php if (isset($item['suggest_' . $feature_key]) && $item['suggest_' . $feature_key] == 1) { echo ' checked="checked"'; } ?> /> <label for="datebook-suggest-<?php echo esc_attr($feature_key); ?>"><?php esc_html_e('Suggest this feature even if it is not included in the plan', 'datebook'); ?></label>

									</td>
								</tr>

								</table>

								<?php }
								
								if (isset($feature_key) && $feature_key == 'simultaneous_chatters') { ?>

								<table cellspacing="0" cellpadding="0" class="form-table" bgcolor="#F7F5EE">

								<tr>
									<td>

									<label for="datebook-quantity_<?php echo esc_attr($feature_key); ?>"><?php esc_html_e('Max. quantity of users to chat', 'datebook'); ?>:</label> 
									<input name="quantity_<?php echo esc_attr($feature_key); ?>" type="text" style="width: 120px" value="<?php if (isset($item['quantity_' . $feature_key]) && $item['quantity_' . $feature_key] > 0) { echo esc_attr($item['quantity_' . $feature_key]); } ?>" size="5" maxlength="5" placeholder="<?php echo esc_html__( 'Enter number', 'datebook' ); ?>" /> <br />
									<em><small><?php esc_html_e('User can send messages (chat) to this quantity of other users at one period? (0 = unlimited)', 'datebook'); ?></small></em>

									</td>
								</tr>
								<tr>
									<td>

									<label for="datebook-period_<?php echo esc_attr($feature_key); ?>"><?php echo esc_html__( 'Period of time for limited chat', 'datebook' ); ?>:</label> 
									<input name="period_number_<?php echo esc_attr($feature_key); ?>" type="text" style="width: 120px" value="<?php if (isset($item['period_number_' . $feature_key])) { echo esc_attr($item['period_number_' . $feature_key]); } ?>" size="3" placeholder="<?php esc_html_e('Enter number', 'datebook'); ?>" />
						
									<?php

									if (isset($item['period_' . $feature_key])) {
										$item_period_feature_key = $item['period_' . $feature_key];
									} else {
										$item_period_feature_key = '';
									}

									$all_periods_position = new DateBook_Register_Plans_Table;
									echo $all_periods_position->get_all_register_plans($item_period_feature_key, 'period_' . $feature_key);

									?>
									
									<br />
									
									<em><small><?php esc_html_e('After this period of time user can send messages again to another quantity of users.', 'datebook'); ?></small></em>

									</td>
								</tr>
								
								<tr class="form-field" style=" <?php if (of_get_option('points_system') == 0) { echo 'display:none'; } ?>">
									<td><?php esc_html_e('Sending 1 message', 'datebook'); ?> <?php esc_html_e('costs', 'datebook'); ?> <input id="datebook-points-sending-one-message" name="points_for_one_message" type="text" style="width: 150px" value="<?php if (isset($item['points_for_one_message'])) { echo esc_attr($item['points_for_one_message']); } else { echo '1'; } ?>" size="5" maxlength="5" class="code" placeholder="<?php echo esc_html__( 'Enter number', 'datebook' ); ?>" /> <?php esc_html_e('points', 'datebook'); ?> (<?php esc_html_e('0 = disabled', 'datebook'); ?>)</td>
								</tr>

								<tr>
									<td>

									<input id="datebook-suggest-<?php echo esc_attr($feature_key); ?>" name="suggest_<?php echo esc_attr($feature_key); ?>" type="checkbox" value="1" <?php if (isset($item['suggest_' . $feature_key]) && $item['suggest_' . $feature_key] == 1) { echo ' checked="checked"'; } ?> /> <label for="datebook-suggest-<?php echo esc_attr($feature_key); ?>"><?php esc_html_e('If you set a limit for the period, then show a message suggesting switching to another plan that has no limits', 'datebook'); ?></label>

									</td>
								</tr>

								</table>

								<?php } else if (isset($feature_key) && $feature_key == 'upload_images') { ?>

									<table cellspacing="0" cellpadding="0" class="form-table" bgcolor="#F7F5EE">
										<tr class="form-field">
											<td><?php esc_html_e('Max. images allowed to upload', 'datebook'); ?> <input id="datebook-limitphotos" name="subscription_limitphotos" type="text" style="width: 100px" value="<?php if (isset($item['subscription_limitphotos'])) { echo esc_attr($item['subscription_limitphotos']); } ?>" size="3" maxlength="5"  placeholder="<?php echo esc_html__( 'Enter number', 'datebook' ); ?>" /> <label for="datebook-limitphotos"><?php echo esc_html__( 'photos', 'datebook' ); ?></label> (<?php esc_html_e('0 = unlimited', 'datebook'); ?>)</td>
										</tr>
							
										<tr class="form-field" style=" <?php if (of_get_option('points_system') == 0) { echo 'display:none'; } ?>">
											<td><?php esc_html_e('Uploading 1 image', 'datebook'); ?> <?php esc_html_e('costs', 'datebook'); ?> <input id="datebook-points-uploading-one-image" name="points_for_one_image" type="text" style="width: 150px" value="<?php if (isset($item['points_for_one_image'])) { echo esc_attr($item['points_for_one_image']); } else { echo '1'; } ?>" size="5" maxlength="5" class="code" placeholder="<?php echo esc_html__( 'Enter number', 'datebook' ); ?>" /> <?php esc_html_e('points', 'datebook'); ?> (<?php esc_html_e('0 = disabled', 'datebook'); ?>)</td>
										</tr>
									</table>

								<?php } else { ?>
								
								<table cellspacing="0" cellpadding="0" class="form-table" bgcolor="#F7F5EE" style=" <?php if (of_get_option('points_system') == 0) { echo 'display:none'; } ?>">
									<tr class="form-field">
										<td><?php esc_html_e($feature_name, 'datebook'); ?> <?php esc_html_e('costs', 'datebook'); ?> <input id="datebook-points-<?php echo esc_attr($feature_key); ?>" name="points_for_<?php echo esc_attr($feature_key); ?>" type="text" style="width: 150px" value="<?php if (isset($item['points_for_' . $feature_key])) { echo esc_attr($item['points_for_' . $feature_key]); } else { echo '5'; } ?>" size="5" maxlength="5" class="code" placeholder="<?php echo esc_html__( 'Enter number', 'datebook' ); ?>" /> <?php esc_html_e('points', 'datebook'); ?> (<?php esc_html_e('0 = disabled', 'datebook'); ?>)</td>
									</tr>
								</table>

								<?php } ?>
								



								<?php if ($item['subscription_type'] == 'feature'){ ?>
								
								<table cellspacing="10" cellpadding="0" bgcolor="#F7F5EE">

									<tr class="form-field">
										<th valign="top" scope="row">
											<label for="datebook-activate-by-default-<?php echo esc_attr($feature_key); ?>"><?php esc_html_e('Activate this feature by default', 'datebook'); ?></label>
										</th>
										<td>
											<input id="datebook-activate-by-default-<?php echo esc_attr($feature_key); ?>"  name="activate_by_default_<?php echo esc_attr($feature_key); ?>" value="1" type="checkbox" <?php if (isset($item['activate_by_default_' . $feature_key]) && $item['activate_by_default_' . $feature_key] == 1) { echo ' checked="checked"'; } ?> /><label for="datebook-activate-by-default-<?php echo esc_attr($feature_key); ?>"><?php esc_html_e('Yes', 'datebook'); ?>. <?php esc_html_e('Member will be subscribed to this feature automatically after registration on the website', 'datebook'); ?>.</label><br>
											<input id="datebook-hide-by-default-<?php echo esc_attr($feature_key); ?>"  name="hide_by_default_<?php echo esc_attr($feature_key); ?>" value="1" type="checkbox" <?php if (isset($item['hide_by_default_' . $feature_key]) && $item['hide_by_default_' . $feature_key] == 1) { echo ' checked="checked"'; } ?> /><label for="datebook-hide-by-default-<?php echo esc_attr($feature_key); ?>"><?php esc_html_e('Hide', 'datebook'); ?>. <?php esc_html_e('Do not show this feature to a member if he subscribed to it by default', 'datebook'); ?>.</label>
										</td>
									</tr>

									<tr class="form-field">
										<th valign="top" scope="row">
											<label for="datebook-hide-after-expire-<?php echo esc_attr($feature_key); ?>"><?php esc_html_e('Hide this feature after expiration', 'datebook'); ?></label>
										</th>
										<td><input id="datebook-hide-after-expire-<?php echo esc_attr($feature_key); ?>"  name="hide_after_expire_<?php echo esc_attr($feature_key); ?>" value="1" type="checkbox" <?php if (isset($item['hide_after_expire_' . $feature_key]) && $item['hide_after_expire_' . $feature_key] == 1) { echo ' checked="checked"'; } ?> /><label for="datebook-hide-after-expire-<?php echo esc_attr($feature_key); ?>"><?php esc_html_e('Yes', 'datebook'); ?>. <?php esc_html_e('If a member has subscribed to this feature, he will not be able to subscribe to this feature a second time after it expires', 'datebook'); ?>.</label></td>
									</tr>

									<tr class="form-field">
										<th valign="top" scope="row">
											<label for="wc_subscription_product_<?php echo esc_attr($feature_key); ?>"><?php esc_html_e('WooCommerce Subscription', 'datebook'); ?></label>
										</th>
										<td>

										<?php

										if ( class_exists( 'WooCommerce' ) ) {

											$item_wc_subscription_product = isset($item['wc_subscription_product_' . $feature_key]) ? $item['wc_subscription_product_' . $feature_key] : '';

											$all_wc_subscription_plans = new DateBook_Register_Plans_Table;
											echo $all_wc_subscription_plans->get_all_wc_subscription_plans($item_wc_subscription_product, 'wc_subscription_product_' . $feature_key);

											echo ' ' . esc_html__('Select a WooCommerce product. It will be connected to this feature.', 'datebook');
											
										} else {

											esc_html_e('WooCommerce is not installed and activated', 'datebook');

										}

										?>

										</td>
									</tr>


									<tr class="form-field">
										<th valign="top" scope="row">
											<label for="datebook-price-<?php echo esc_attr($feature_key); ?>"><?php echo esc_html__( 'Price', 'datebook' ); ?> + <?php esc_html_e('Period', 'datebook'); ?></label>
											<br />
											<small><?php esc_html_e('What currency is used you can look at the Dashboard -> Payment tab', 'datebook'); ?></small>
										</th>
										<td>
											<table id="datebook-pricetable" cellspacing="0" cellpadding="0" class="form-table" style="width:100%" bgcolor="#EDEADE">
												<tr>
													<td>

													<label for="datebook-price-<?php echo esc_attr($feature_key); ?>"><?php echo esc_html__( 'Price', 'datebook' ); ?>:</label> &nbsp; 
													<input id="datebook-price-<?php echo esc_attr($feature_key); ?>" name="price_<?php echo esc_attr($feature_key); ?>" type="text" style="width: 120px" value="<?php if (isset($item['price_' . $feature_key])) { echo esc_attr($item['price_' . $feature_key]); } ?>" size="5" maxlength="10" class="code" placeholder="<?php echo esc_html__( 'Enter number', 'datebook' ); ?>" required="required" /> <?php esc_html_e('Enter ONLY number', 'datebook'); ?>

													<div class="clear"></div>

													<label for="datebook-period-<?php echo esc_attr($feature_key); ?>"><?php echo esc_html__( 'Period', 'datebook' ); ?>:</label> 
													<input id="register_period_number_<?php echo esc_attr($feature_key); ?>" name="register_period_number_<?php echo esc_attr($feature_key); ?>" type="text" style="width: 120px" value="<?php if (isset($item['register_period_number_' . $feature_key])) { echo esc_attr($item['register_period_number_' . $feature_key]); } ?>" size="3" class="code" placeholder="<?php esc_html_e('Enter number', 'datebook'); ?>" required="required" />

													<?php

													if (isset($item['register_period_' . $feature_key])) {
														$item_register_period = esc_attr($item['register_period_' . $feature_key]);
													} else {
														$item_register_period = '';
													}

													$all_register_plans_register_periods = new DateBook_Register_Plans_Table;
													echo $all_register_plans_register_periods->get_all_register_plans($item_register_period, 'register_period_' . $feature_key);

													?>

													</td>
												</tr>

										</table>
									</td>
								</tr>

								<tr class="form-field" style=" <?php if (of_get_option('points_system') == 0) { echo 'display:none'; } ?>">
									<th valign="top" scope="row">
										<label for="datebook-pointsquantity-<?php echo esc_attr($feature_key); ?>"><?php echo esc_html__( 'Points', 'datebook' ); ?></label>
									</th>
									<td><input id="datebook-pointsquantity-<?php echo esc_attr($feature_key); ?>" name="pointsquantity_<?php echo esc_attr($feature_key); ?>" type="text" style="width: 150px" value="<?php if (isset($item['pointsquantity_' . $feature_key])) { echo esc_attr($item['pointsquantity_' . $feature_key]); } else { echo '50'; } ?>" size="5" maxlength="5" class="code" placeholder="<?php echo esc_html__( 'Enter number', 'datebook' ); ?>" /> <?php esc_html_e('How many points will be automatically added after paying for this feature?', 'datebook'); ?>

									</td>
								</tr>

								<tr class="form-field">
									<th valign="top" scope="row">
										<label for="recurring-<?php echo esc_attr($feature_key); ?>"><?php esc_html_e('Recurring payment', 'datebook'); ?></label>
									</th>
									<td>

									<?php

									if (isset($item['recurring_' . $feature_key])) {
										$item_recurring = esc_attr($item['recurring_' . $feature_key]);
									} else {
										$item_recurring = 0;
									}

									$all_recurring_select = new DateBook_Register_Plans_Table;
									echo $all_recurring_select->get_recurring_select($item_recurring, 'recurring_' . $feature_key);

									?>

				                    </td>
								</tr>

								<tr class="form-field">
									<th valign="top" scope="row">
										<label for="recurring-<?php echo esc_attr($feature_key); ?>"><?php esc_html_e('Free for genders', 'datebook'); ?></label>
									</th>
									<td>

									<p><?php esc_html_e('Select genders who must not pay for this feature. They will be able to activate this feature for free.', 'datebook'); ?></p>

									<?php

									global $datebook_gender_titles;

									// get genders
									$sex_titles_array = array();

									foreach ( $datebook_gender_titles as $key => $value ) {

										//$sex_titles_array[$key] = esc_html__( $value, 'datebook' );

										?>

										<input id="datebook-free-for-gender-<?php echo esc_attr($feature_key); ?>-<?php echo esc_attr($key); ?>" name="free_for_gender_<?php echo esc_attr($feature_key); ?>[<?php echo esc_attr($key); ?>]" type="checkbox" value="1" <?php if (isset($item['free_for_gender_' . $feature_key][$key]) && $item['free_for_gender_' . $feature_key][$key] == 1) { echo ' checked="checked"'; } ?> /> <label for="datebook-free-for-gender-<?php echo esc_attr($feature_key); ?>-<?php echo esc_attr($key); ?>"><?php esc_html_e($value, 'datebook'); ?></label><br>

									<?php } ?>

					                    </td>
									</tr>

								</table>

								<?php } ?>


								</td>
							</tr>

							<?php } ?>


						</table>
					</td>
				</tr>

			</tbody>
		</table>

	<?php

	}


	/**
	 * Simple function that validates data and retrieve bool on success
	 * and error message(s) on error
	 *
	 * @param $item
	 * @return bool|string
	 */
	static function validate_data($item) {

		$data = array();
		$messages = array();

		if(is_array($item)) {

			if (empty($item['name'])) $messages[] = esc_html__('Title is required', 'datebook');

			if(empty($item['register_period'])) $messages[] = esc_html__('Registration period is required', 'datebook');
		
			if(!empty($item['register_period']) && ($item['register_period'] == 'radio' || $item['register_period'] == 'checkbox' || $item['register_period'] == 'select') && empty($item['field_items'])) $messages[] = esc_html__('Field Items are required', 'datebook');

			$data['form_fields'] = $item;

			if (is_array($data)) {

				$data['success'] = true;
				
			} else {

				$data['success'] = false;
				$data['error_message'] = esc_html__( 'Oops! Something went wrong.', 'datebook' );

			}

		} else {

			$data['success'] = false;
			$data['error_message'] = esc_html__( 'Oops! Something went wrong.', 'datebook' );

		}

		return $data;

	}

}