<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class DateBook_Post_Type_Profile
 *
 * @class DateBook_Post_Type_Profile
 * @package DateBook/Classes/Post_Types
 * @author Pagick Themes
 */
class DateBook_Post_Type_Profile {


	/**
	 * Hook actions and filters.
	 * 
	 * @since 1.0
	 * @access public
	 */
	public static function init() {

		add_post_type_support( DATEBOOK_POST_TYPE, 'thumbnail' );
		add_action( 'init', array( __CLASS__, 'datebook_definition' ) );
		add_action( 'init', array( __CLASS__, 'datebook_set_subscriber_permissions' ) );


		add_action( 'profiletype_add_form_fields', array( __CLASS__, 'process_add_category' ), 10, 2);
		add_action( 'created_profiletype', array ( __CLASS__, 'save_category_image' ), 10, 2 );
		add_action( 'profiletype_edit_form_fields', array( __CLASS__, 'process_edit_category'), 10, 2);

		add_action( 'edited_profiletype', array( __CLASS__, 'process_save_category' ), 10, 2);
		add_action( 'create_profiletype', array( __CLASS__, 'process_save_category' ));

		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'load_media' ) );

	}


	/**
	 * Load media javascript files
	 *
	 * @access public
	 * @return void
	 */
	public static function load_media() {
		wp_enqueue_media();
	}


	/**
	 * Custom post type definition
	 *
	 * @access public
	 * @return void
	 */
	public static function datebook_definition() {
		
		if(of_get_option('what_pagebuilder_for_profile') == '2'){
			if(of_get_option('page_created_for_profile') > 0){
				$slug_rewritten = get_post_field('post_name', of_get_option('page_created_for_profile'));
			} else {
				$slug_rewritten = of_get_option('profile_permalink_slug');
			}
		} else if(of_get_option('profile_permalink_slug')){
			$slug_rewritten = DateBook_Utilities::datebook_makeslug(of_get_option('profile_permalink_slug'));
		} else {
			$slug_rewritten = DATEBOOK_POST_TYPE;
		}

		$post_types = array(
			array(esc_html__( 'Profiles', 'datebook' ), $slug_rewritten, 'dashicons-admin-users', esc_html__( 'Add personal profile', 'datebook' ))
		);
		
		foreach ($post_types as $t) {
			$args = array(
				'label' => $t[0],
				'description ' => $t[3],
				'public' => true,
				'publicly_queryable' => true,
				'exclude_from_search' => true,
				'show_ui' => true,
				'show_in_rest' => true,
				'rest_controller_class' => 'WP_REST_Posts_Controller',
				'show_in_menu' => true,
				'menu_position' => 51,
				'hierarchical' => false,
				'menu_icon' => $t[2], // dashicons-location, dashicons-format-chat, dashicons-admin-comments, dashicons-email-alt, dashicons-pressthis, dashicons-megaphone
				'has_archive' => false,
				'query_var' => true,
				'rewrite' => array('slug' => $t[1]),
				'supports' => array('title','editor','author','excerpt','custom-fields','comments', 'thumbnail'),
				'capabilities' => array(
					'create_posts' => false
				),
				'map_meta_cap' => true
			);
			register_post_type( DATEBOOK_POST_TYPE, $args );
		}
		
		$tax_slug_rewritten = of_get_option('profiles_tax_permalink_slug') ? DateBook_Utilities::datebook_makeslug(of_get_option('profiles_tax_permalink_slug')) : 'profiletype';

		register_taxonomy(DATEBOOK_PROFILES_TAX, DATEBOOK_POST_TYPE, array(
				'hierarchical' => true,
				'label' => esc_html__( 'Types', 'datebook' ),
				'sort' => true,
				'show_ui' => true,
				'show_admin_column' => true,
				'query_var' => true,
				'args' => array( 'orderby' => 'term_order' ),
				'rewrite' => array( 'slug' => $tax_slug_rewritten )
			)
		);

	}


	/**
	 * Allow subscribers to upload images
	 *
	 * @access public
	 * @return void
	 */
	public static function datebook_set_subscriber_permissions( ) {

		$subscriber = get_role( 'subscriber' );
		$subscriber->add_cap( 'upload_files' );
		$subscriber->add_cap( 'delete_posts' );
		$subscriber->add_cap( 'edit_post' );

	}


	/**
	 * Get languages
	 *
	 * @access public
	 * @return void
	 */
	private static function get_languages(){

		global $wp_locale_all;

		// get languages
		$languages_array = array();

		$languages_array['en_US'] = esc_html__( 'English', 'datebook' );

		$languages = DateBook_Localization::datebook_all_available_language_files();

		foreach($languages as $language) {
			if($language['checked'] == 1){
				$languages_array[$language['locale']] = $wp_locale_all[$language['locale']];
			}
		}
		
		return $languages_array;
	}


	/**
	 * Add custom taxonomy - Multilingual category
	 *
	 * @access public
	 * @return void
	 */
	public static function process_add_category( $term ) {

		// Get profile form array
		$get_profile_form_array = get_option('datebook_profile_form');

		if(count($get_profile_form_array) > 0) {

		?>

			<div class="form-field term-parent-wrap">
				<label for="parent"><?php echo esc_html__('Select the Registration form for this type of profile', 'datebook'); ?></label>
				<select name="registration_formname" id="parent" class="postform">

					<?php

					foreach ( $get_profile_form_array as $key => $value ) {
						
						$formtitle = isset($value['formtitle']) && !empty($value['formtitle']) ? $value['formtitle'] : $key;

						echo '<option class="level-0" value="' . $key . '">' . $formtitle . '</option>';

					}

					?>
				</select>

			</div>
			
			<div class="form-field term-parent-wrap">
				<label for="parent"><?php echo esc_html__('Select the Settings form for this type of profile', 'datebook'); ?></label>
				<select name="settings_formname" id="parent" class="postform">

					<?php

					foreach ( $get_profile_form_array as $key => $value ) {
						
						$formtitle = isset($value['formtitle']) && !empty($value['formtitle']) ? $value['formtitle'] : $key;

						echo '<option class="level-0" value="' . $key . '">' . $formtitle . '</option>';

					}

					?>
				</select>

			</div>
			
			<div class="form-field term-parent-wrap">
				<label for="parent"><?php echo esc_html__('Select the Profile form for this type of profile', 'datebook'); ?></label>
				<select name="profile_formname" id="parent" class="postform">

					<?php

					foreach ( $get_profile_form_array as $key => $value ) {
						
						$formtitle = isset($value['formtitle']) && !empty($value['formtitle']) ? $value['formtitle'] : $key;

						echo '<option class="level-0" value="' . $key . '">' . $formtitle . '</option>';

					}

					?>
				</select>

			</div>

		<?php
		
		}

		$languages_array = self::get_languages();

		foreach ( $languages_array as $language_key => $language_name ) {

		?>

			<div class="form-field">
				<label for="taxProfileCategory_<?php echo esc_attr($language_key); ?>"><?php echo sprintf( esc_html__( 'Type (%1$s)', 'datebook' ), $language_name ); ?></label><input type="text" name="taxProfileCategory_<?php echo esc_attr($language_key); ?>" id="taxProfileCategory_<?php echo esc_attr($language_key); ?>" value="" />
			</div>

		<?php } ?>

		<div class="form-field term-group">
			<label for="datebook-profiletype-image-id"><?php echo esc_html__('Image', 'datebook'); ?></label>
			<input type="hidden" id="datebook-profiletype-image-id" name="datebook-profiletype-image-id" class="custom_media_url" value="">
			<div id="datebook-profiletype-image-wrapper"></div>

			<p><input type="button" class="button button-secondary ct_tax_media_button" id="ct_tax_media_button" name="ct_tax_media_button" value="<?php echo esc_html__( 'Add Image', 'datebook' ); ?>" />
			<input type="button" class="button button-secondary ct_tax_media_remove" id="ct_tax_media_remove" name="ct_tax_media_remove" value="<?php echo esc_html__( 'Remove Image', 'datebook' ); ?>" /></p>
		</div>

		<?php
	}


	/*
	 * Save the image field
	 * @since 4.5.6
	 */
	public static function save_category_image ( $term_id, $tt_id ) {
		if( isset( $_POST['datebook-profiletype-image-id'] ) && '' !== $_POST['datebook-profiletype-image-id'] ){
			$image = $_POST['datebook-profiletype-image-id'];
			add_term_meta( $term_id, 'datebook-profiletype-image-id', $image, true );
		}
	}


	/**
	 * Edit custom taxonomy - Multilingual category
	 *
	 * @access public
	 * @return void
	 */
	public static function process_edit_category( $term ) {
		
		// put the term ID into a variable
		$t_id = $term->term_id;

		// Get profile form array
		$get_profile_form_array = get_option('datebook_profile_form');

		if(count($get_profile_form_array) > 0) {

		?>

		<tr class="form-field term-group-wrap">
			<th scope="row">
				<label for="parent"><?php echo esc_html__('Select the Registration form for this type of profile', 'datebook'); ?></label>
			</th>
			<td>
				<select name="registration_formname" id="parent" class="postform">

					<?php

					foreach ( $get_profile_form_array as $key => $value ) {
						
						$term_category = get_term_meta( $t_id, 'registration_formname', true );
						
						$formtitle = isset($value['formtitle']) && !empty($value['formtitle']) ? $value['formtitle'] : $key;

						$selected_category = $term_category == $key ? 'selected' : '';

						echo '<option class="level-0" value="' . $key . '" ' . $selected_category . '>' . $formtitle . '</option>';

					}

					?>
				</select>
			</td>
		</tr>
		
		<tr class="form-field term-group-wrap">
			<th scope="row">
				<label for="parent"><?php echo esc_html__('Select the Settings form for this type of profile', 'datebook'); ?></label>
			</th>
			<td>
				<select name="settings_formname" id="parent" class="postform">

					<?php

					foreach ( $get_profile_form_array as $key => $value ) {
						
						$term_category = get_term_meta( $t_id, 'settings_formname', true );
						
						$formtitle = isset($value['formtitle']) && !empty($value['formtitle']) ? $value['formtitle'] : $key;

						$selected_category = $term_category == $key ? 'selected' : '';

						echo '<option class="level-0" value="' . $key . '" ' . $selected_category . '>' . $formtitle . '</option>';

					}

					?>
				</select>
			</td>
		</tr>
		
		<tr class="form-field term-group-wrap">
			<th scope="row">
				<label for="parent"><?php echo esc_html__('Select the Profile form for this type of profile', 'datebook'); ?></label>
			</th>
			<td>
				<select name="profile_formname" id="parent" class="postform">

					<?php

					foreach ( $get_profile_form_array as $key => $value ) {
						
						$term_category = get_term_meta( $t_id, 'profile_formname', true );
						
						$formtitle = isset($value['formtitle']) && !empty($value['formtitle']) ? $value['formtitle'] : $key;

						$selected_category = $term_category == $key ? 'selected' : '';

						echo '<option class="level-0" value="' . $key . '" ' . $selected_category . '>' . $formtitle . '</option>';

					}

					?>
				</select>
			</td>
		</tr>

		<?php

		}

		
		$languages_array = self::get_languages();

		foreach ( $languages_array as $language_key => $language_name ) {

			$term_category = get_term_meta( $t_id, 'datebook_profiletype_' . esc_attr($language_key), true );

			?>

			<tr class="form-field">
				<th><label for="taxProfileCategory_<?php echo esc_attr($language_key); ?>"><?php echo sprintf( esc_html__( 'Type (%1$s)', 'datebook' ), $language_name ); ?></label></th>
				<td><input type="text" name="taxProfileCategory_<?php echo esc_attr($language_key); ?>" id="taxProfileCategory_<?php echo esc_attr($language_key); ?>" value="<?php echo esc_attr( $term_category ) ? esc_attr( $term_category ) : ''; ?>"></td>
			</tr>

			<?php

		}
		
		?>

		<tr class="form-field term-group-wrap">
			<th scope="row">
				<label for="datebook-profiletype-image-id"><?php echo esc_html__( 'Image', 'datebook' ); ?></label>
			</th>
			<td>
			<?php $image_id = get_term_meta ( $term -> term_id, 'datebook-profiletype-image-id', true ); ?>
			<input type="hidden" id="datebook-profiletype-image-id" name="datebook-profiletype-image-id" value="<?php echo $image_id; ?>">
			<div id="datebook-profiletype-image-wrapper">
				<?php

				if ( $image_id ) {
					echo wp_get_attachment_image ( $image_id, 'thumbnail' );
				}

				?>
			</div>
			<p><input type="button" class="button button-secondary ct_tax_media_button" id="ct_tax_media_button" name="ct_tax_media_button" value="<?php echo esc_html__( 'Add Image', 'datebook' ); ?>" />
			<input type="button" class="button button-secondary ct_tax_media_remove" id="ct_tax_media_remove" name="ct_tax_media_remove" value="<?php echo esc_html__( 'Remove Image', 'datebook' ); ?>" /></p>
			</td>
		</tr>

		<?php

	}


	/**
	 * Saving custom taxonomy - Multilingual category
	 *
	 * @access public
	 * @return void
	 */
	public static function process_save_category( $term_id ) {
		
		if( isset( $_POST['registration_formname'] ) && '' !== $_POST['registration_formname'] ){
			$registration_formname = $_POST['registration_formname'];
			update_term_meta( $term_id, 'registration_formname', $registration_formname );
		} else {
			update_term_meta( $term_id, 'registration_formname', '' );
		}

		if( isset( $_POST['settings_formname'] ) && '' !== $_POST['settings_formname'] ){
			$settings_formname = $_POST['settings_formname'];
			update_term_meta( $term_id, 'settings_formname', $settings_formname );
		} else {
			update_term_meta( $term_id, 'settings_formname', '' );
		}

		if( isset( $_POST['profile_formname'] ) && '' !== $_POST['profile_formname'] ){
			$profile_formname = $_POST['profile_formname'];
			update_term_meta( $term_id, 'profile_formname', $profile_formname );
		} else {
			update_term_meta( $term_id, 'profile_formname', '' );
		}

		$languages_array = self::get_languages();

		foreach ( $languages_array as $language_key => $language_name ) {

			if ( isset( $_POST['taxProfileCategory_' . $language_key] ) ) {
				$term_category = $_POST['taxProfileCategory_' . $language_key];
				if( $term_category ) {
					 update_term_meta( $term_id, 'datebook_profiletype_' . esc_attr($language_key), esc_attr($term_category) );
				}
			}
		}

		if( isset( $_POST['datebook-profiletype-image-id'] ) && '' !== $_POST['datebook-profiletype-image-id'] ){
			$image = $_POST['datebook-profiletype-image-id'];
			update_term_meta( $term_id, 'datebook-profiletype-image-id', $image );
		} else {
			update_term_meta( $term_id, 'datebook-profiletype-image-id', '' );
		}

	}

}

DateBook_Post_Type_Profile::init();