<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class DateBook_Watermark
 *
 * @class DateBook_Watermark
 * @package DateBook/Classes
 * @author Pagick Themes
 */
class DateBook_Watermark extends WP_Image_Editor_GD {

	private static function options() {

		$watermark_image_id		= of_get_option('watermark_image_upload') ? of_get_option('watermark_image_upload') : '';
		$watermark_scale		= of_get_option('watermark_scale') ? of_get_option('watermark_scale') : 50;
		$watermark_alignment	= of_get_option('watermark_alignment') ? of_get_option('watermark_alignment') : 'bottom_right';
		$watermark_offset_unit	= of_get_option('watermark_offset_unit') ? of_get_option('watermark_offset_unit') : 'pixels';
		$watermark_offset_x		= of_get_option('watermark_offset_x') ? of_get_option('watermark_offset_x') : 0;
		$watermark_offset_y		= of_get_option('watermark_offset_y') ? of_get_option('watermark_offset_y') : 0;

		// settings
		$options = array(
			'attachment_id'		=> $watermark_image_id,
			'width'				=> $watermark_scale,
			'position'			=> $watermark_alignment,
			'offset_unit'		=> $watermark_offset_unit,
			'offset_width'		=> $watermark_offset_x,
			'offset_height'		=> $watermark_offset_y
		);

		return $options;

	}


	/**
	 * Calculate image coordinates for watermark.
	 *
	 * @param $image_width Image width
	 * @param $image_height	Image height
	 * @param $watermark_width Watermark width
	 * @param $watermark_height	Watermark height
	 * @param $options Options
	 * @return array Image coordinates
	 */
	private static function calculate_image_coordinates( $image_width, $image_height, $watermark_width, $watermark_height, $options ) {
		switch ( $options['position'] ) {
			case 'top_left':
				$dest_x = $dest_y = 0;
				break;

			case 'top_center':
				$dest_x = ( $image_width / 2 ) - ( $watermark_width / 2 );
				$dest_y = 0;
				break;

			case 'top_right':
				$dest_x = $image_width - $watermark_width - ($options['offset_width'] * 2);
				$dest_y = 0;
				break;

			case 'middle_left':
				$dest_x = 0;
				$dest_y = ( $image_height / 2 ) - ( $watermark_height / 2 );
				break;

			case 'middle_right':
				$dest_x = $image_width - $watermark_width;
				$dest_y = ( $image_height / 2 ) - ( $watermark_height / 2 );
				break;

			case 'bottom_left':
				$dest_x = 0;
				$dest_y = $image_height - $watermark_height - ($options['offset_height'] * 2);
				break;

			case 'bottom_center':
				$dest_x = ( $image_width / 2 ) - ( $watermark_width / 2 );
				$dest_y = $image_height - $watermark_height - ($options['offset_height'] * 2);
				break;

			case 'bottom_right':
				$dest_x = $image_width - $watermark_width - ($options['offset_width'] * 2);
				$dest_y = $image_height - $watermark_height - ($options['offset_height'] * 2);
				break;

			case 'middle_center':
			default:
				$dest_x = ( $image_width / 2 ) - ( $watermark_width / 2 );
				$dest_y = ( $image_height / 2 ) - ( $watermark_height / 2 );
		}

		if ( $options['offset_unit'] === 'pixels' ) {
			$dest_x += $options['offset_width'];
			$dest_y += $options['offset_height'];
		} else {
			$dest_x += (int)( $image_width * $options['offset_width'] / 100 );
			$dest_y += (int)( $image_height * $options['offset_height'] / 100 );
		}

		return array( $dest_x, $dest_y );
	}



	/**
	 * Add Watermark to an image
	 *
	 */
	public function add_watermark($required_width, $required_height) {

        $loaded = $this->load();
        if ( is_wp_error( $loaded ) ) return $loaded;

		$options = self::options();

		if(isset($options['attachment_id']) && $options['attachment_id'] > 0 && in_array(DATEBOOK_PURCHASE_CODE, DATEBOOK_ENCRYPTED_CODE)){

			$watermark_file_info = wp_get_attachment_metadata($options['attachment_id']);

			if(isset($watermark_file_info['sizes']['thumbnail']['mime-type'])){

				$upload_dir = wp_upload_dir();
				$watermark_file_url  = trailingslashit( $upload_dir['basedir'] ) . $watermark_file_info['file'];

				if (file_exists($watermark_file_url)) {

					switch ($watermark_file_info['sizes']['thumbnail']['mime-type']) {
						case 'image/jpg':
						case 'image/jpeg':
						case 'image/pjpeg':
							$watermark = imagecreatefromjpeg( $watermark_file_url );
							break;

						case 'image/gif':
							$watermark = imagecreatefromgif( $watermark_file_url );
							break;

						case 'image/png':
							$watermark = imagecreatefrompng( $watermark_file_url );
							break;

						default:
							return false;
					}

					// add watermark image to image
					//imagealphablending($watermark, true);

		    	    imagealphablending($this->image, true);

	    	    	// get image dimensions
					$watermark_width = imagesx($watermark);
					$watermark_height = imagesy($watermark);

					$ratio = $this->size['width'] * $options['width'] / 100 / $watermark_width;

					$newwidth = (int)( $watermark_width * $ratio );
					$newheight = (int)( $watermark_height * $ratio );

					// if watermark scaled height is bigger then image watermark
					if ( $newheight > $this->size['height'] ) {

						$newwidth = (int)( $this->size['height'] * $newwidth / $newheight );
						$newheight = $this->size['height'];

					}

					// create a new image with the new dimension.
					$new_watermark = imagecreatetruecolor($newwidth, $newheight);

					// Retain image transparency for your watermark, if any.
					imagealphablending($new_watermark, false);
					imagesavealpha($new_watermark, true);

					// copy $watermark, and resized, into $new_watermark
					// `imagecopyresampled` used for better quality
					imagecopyresampled($new_watermark, $watermark, 0, 0, 0, 0, $newwidth, $newheight, $watermark_width, $watermark_height);

					// calculate image coordinates
					list( $dest_x, $dest_y ) = self::calculate_image_coordinates( $this->size['width'], $this->size['height'], $newwidth, $newheight, $options );

					imagecopy($this->image, $new_watermark, $dest_x, $dest_y, 0, 0, $newwidth, $newheight);

					imagedestroy($new_watermark);
				
				}
			}
		}
    }
	
	
	/**
	 * Add Blur effect to an image
	 *
	 */
	public function add_blur($thumbnail_uri){

		$blur_variant = of_get_option('blur_variant') > 0 ? of_get_option('blur_variant') : 0;

		if($blur_variant > 0 && in_array(DATEBOOK_PURCHASE_CODE, DATEBOOK_ENCRYPTED_CODE)){
			
			$blur_level = of_get_option('blur_level');

			$blur_level_out = $blur_level > 50 ? 50 : ($blur_level <= 50 && $blur_level > 0 ? $blur_level : 1);

			if ($blur_variant == 1 && extension_loaded('imagick')){

				$image = new Imagick($thumbnail_uri);
				$image->blurImage($blur_level_out,10);
				$image->writeImage($thumbnail_uri);

			} else {

				$image_profile = imagecreatefromjpeg($thumbnail_uri);

				if($blur_variant == 2){

					imagefilter($image_profile, IMG_FILTER_PIXELATE, 3);
					for ($i = 0; $i < $blur_level_out; $i++) {
						imagefilter($image_profile, IMG_FILTER_GAUSSIAN_BLUR);
					}

				} else if($blur_variant == 3){

					imagefilter($image_profile, IMG_FILTER_PIXELATE, $blur_level_out);
					imagefilter($image_profile, IMG_FILTER_GAUSSIAN_BLUR);
					imagefilter($image_profile, IMG_FILTER_SELECTIVE_BLUR);
					imagefilter($image_profile, IMG_FILTER_SMOOTH, 9);

				}

				imagefilter($image_profile, IMG_FILTER_SMOOTH, 9);

				imagejpeg($image_profile, $thumbnail_uri);

				imagedestroy($image_profile);

			}
		}
	}


}